#include <cstring>
#include <utility>
#include <gromox/exmdb_client.hpp>
#include <gromox/exmdb_rpc.hpp>
#include <gromox/clock.hpp>
#include <gromox/exmdb_common_util.hpp>
#include <gromox/exmdb_ext.hpp>
#include <gromox/exmdb_provider_client.hpp>
#include <gromox/exmdb_server.hpp>
using namespace gromox;
extern unsigned int g_exrpc_debug;
unsigned int g_exrpc_debug;

static void lpc_log(bool ok, const char *dir, const char *func, gromox::time_point tstart, gromox::time_point tend)
{
	if (g_exrpc_debug >= 2 || (!ok && g_exrpc_debug == 1))
		mlog(LV_DEBUG, "LPC %s %s %5luµs %s", dir, !ok ? "ERR" : "ok ",
			static_cast<unsigned long>(std::chrono::duration_cast<std::chrono::microseconds>(tend - tstart).count()), func);
}

BOOL exmdb_client_local::ping_store(const char *dir)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::ping_store(dir);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::ping_store(dir);
	lpc_log(xbresult, dir, "ping_store", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_all_named_propids(const char *dir, PROPID_ARRAY *propids)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_all_named_propids(dir, propids);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_all_named_propids(dir, propids);
	lpc_log(xbresult, dir, "get_all_named_propids", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_named_propids(const char *dir, BOOL b_create, const PROPNAME_ARRAY *ppropnames, PROPID_ARRAY *propids)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_named_propids(dir, b_create, ppropnames, propids);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_named_propids(dir, b_create, ppropnames, propids);
	lpc_log(xbresult, dir, "get_named_propids", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_named_propnames(const char *dir, const PROPID_ARRAY &ppropids, PROPNAME_ARRAY *propnames)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_named_propnames(dir, ppropids, propnames);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_named_propnames(dir, ppropids, propnames);
	lpc_log(xbresult, dir, "get_named_propnames", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_mapping_guid(const char *dir, uint16_t replid, BOOL *b_found, GUID *guid)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_mapping_guid(dir, replid, b_found, guid);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_mapping_guid(dir, replid, b_found, guid);
	lpc_log(xbresult, dir, "get_mapping_guid", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_mapping_replid(const char *dir, GUID guid, uint16_t *replid, ec_error_t *e_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_mapping_replid(dir, guid, replid, e_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_mapping_replid(dir, guid, replid, e_result);
	lpc_log(xbresult, dir, "get_mapping_replid", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_store_all_proptags(const char *dir, PROPTAG_ARRAY *proptags)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_store_all_proptags(dir, proptags);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_store_all_proptags(dir, proptags);
	lpc_log(xbresult, dir, "get_store_all_proptags", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_store_properties(const char *dir, cpid_t cpid, proptag_cspan pproptags, TPROPVAL_ARRAY *propvals)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_store_properties(dir, cpid, pproptags, propvals);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_store_properties(dir, cpid, pproptags, propvals);
	lpc_log(xbresult, dir, "get_store_properties", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::set_store_properties(const char *dir, cpid_t cpid, const TPROPVAL_ARRAY *ppropvals, PROBLEM_ARRAY *problems)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::set_store_properties(dir, cpid, ppropvals, problems);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::set_store_properties(dir, cpid, ppropvals, problems);
	lpc_log(xbresult, dir, "set_store_properties", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::remove_store_properties(const char *dir, proptag_cspan pproptags)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::remove_store_properties(dir, pproptags);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::remove_store_properties(dir, pproptags);
	lpc_log(xbresult, dir, "remove_store_properties", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_mbox_perm(const char *dir, const char *username, uint32_t *permission)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_mbox_perm(dir, username, permission);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_mbox_perm(dir, username, permission);
	lpc_log(xbresult, dir, "get_mbox_perm", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_folder_by_class(const char *dir, const char *str_class, uint64_t *id, std::string *str_explicit)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_folder_by_class(dir, str_class, id, str_explicit);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_folder_by_class(dir, str_class, id, str_explicit);
	lpc_log(xbresult, dir, "get_folder_by_class", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::set_folder_by_class(const char *dir, uint64_t folder_id, const char *str_class, BOOL *b_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::set_folder_by_class(dir, folder_id, str_class, b_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::set_folder_by_class(dir, folder_id, str_class, b_result);
	lpc_log(xbresult, dir, "set_folder_by_class", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_folder_class_table(const char *dir, TARRAY_SET *table)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_folder_class_table(dir, table);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_folder_class_table(dir, table);
	lpc_log(xbresult, dir, "get_folder_class_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::is_folder_present(const char *dir, uint64_t folder_id, BOOL *b_exist)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::is_folder_present(dir, folder_id, b_exist);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::is_folder_present(dir, folder_id, b_exist);
	lpc_log(xbresult, dir, "is_folder_present", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::is_folder_deleted(const char *dir, uint64_t folder_id, BOOL *b_del)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::is_folder_deleted(dir, folder_id, b_del);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::is_folder_deleted(dir, folder_id, b_del);
	lpc_log(xbresult, dir, "is_folder_deleted", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_folder_by_name(const char *dir, uint64_t parent_id, const char *str_name, uint64_t *folder_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_folder_by_name(dir, parent_id, str_name, folder_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_folder_by_name(dir, parent_id, str_name, folder_id);
	lpc_log(xbresult, dir, "get_folder_by_name", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_folder_perm(const char *dir, uint64_t folder_id, const char *username, uint32_t *permission)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_folder_perm(dir, folder_id, username, permission);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_folder_perm(dir, folder_id, username, permission);
	lpc_log(xbresult, dir, "get_folder_perm", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::create_folder_v1(const char *dir, cpid_t cpid, const TPROPVAL_ARRAY *pproperties, uint64_t *folder_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::create_folder_v1(dir, cpid, pproperties, folder_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::create_folder_v1(dir, cpid, pproperties, folder_id);
	lpc_log(xbresult, dir, "create_folder_v1", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::create_folder(const char *dir, cpid_t cpid, const TPROPVAL_ARRAY *pproperties, uint64_t *folder_id, ec_error_t *e_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::create_folder(dir, cpid, pproperties, folder_id, e_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::create_folder(dir, cpid, pproperties, folder_id, e_result);
	lpc_log(xbresult, dir, "create_folder", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_folder_all_proptags(const char *dir, uint64_t folder_id, PROPTAG_ARRAY *proptags)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_folder_all_proptags(dir, folder_id, proptags);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_folder_all_proptags(dir, folder_id, proptags);
	lpc_log(xbresult, dir, "get_folder_all_proptags", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_folder_properties(const char *dir, cpid_t cpid, uint64_t folder_id, proptag_cspan pproptags, TPROPVAL_ARRAY *propvals)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_folder_properties(dir, cpid, folder_id, pproptags, propvals);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_folder_properties(dir, cpid, folder_id, pproptags, propvals);
	lpc_log(xbresult, dir, "get_folder_properties", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::set_folder_properties(const char *dir, cpid_t cpid, uint64_t folder_id, const TPROPVAL_ARRAY *pproperties, PROBLEM_ARRAY *problems)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::set_folder_properties(dir, cpid, folder_id, pproperties, problems);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::set_folder_properties(dir, cpid, folder_id, pproperties, problems);
	lpc_log(xbresult, dir, "set_folder_properties", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::remove_folder_properties(const char *dir, uint64_t folder_id, proptag_cspan pproptags)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::remove_folder_properties(dir, folder_id, pproptags);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::remove_folder_properties(dir, folder_id, pproptags);
	lpc_log(xbresult, dir, "remove_folder_properties", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::delete_folder(const char *dir, cpid_t cpid, uint64_t folder_id, BOOL b_hard, BOOL *b_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::delete_folder(dir, cpid, folder_id, b_hard, b_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::delete_folder(dir, cpid, folder_id, b_hard, b_result);
	lpc_log(xbresult, dir, "delete_folder", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::empty_folder(const char *dir, cpid_t cpid, const char *username, uint64_t folder_id, uint32_t flags, BOOL *b_partial)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::empty_folder(dir, cpid, username, folder_id, flags, b_partial);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::empty_folder(dir, cpid, username, folder_id, flags, b_partial);
	lpc_log(xbresult, dir, "empty_folder", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::is_descendant_folder(const char *dir, uint64_t parent_fid, uint64_t child_fid, BOOL *b_included)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::is_descendant_folder(dir, parent_fid, child_fid, b_included);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::is_descendant_folder(dir, parent_fid, child_fid, b_included);
	lpc_log(xbresult, dir, "is_descendant_folder", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::copy_folder_internal(const char *dir, cpid_t cpid, BOOL b_guest, const char *username, uint64_t src_fid, BOOL b_normal, BOOL b_fai, BOOL b_sub, uint64_t dst_fid, BOOL *b_collid, BOOL *b_partial)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::copy_folder_internal(dir, cpid, b_guest, username, src_fid, b_normal, b_fai, b_sub, dst_fid, b_collid, b_partial);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::copy_folder_internal(dir, cpid, b_guest, username, src_fid, b_normal, b_fai, b_sub, dst_fid, b_collid, b_partial);
	lpc_log(xbresult, dir, "copy_folder_internal", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_search_criteria(const char *dir, uint64_t folder_id, uint32_t *search_status, RESTRICTION **prestriction, EID_ARRAY *folder_ids)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_search_criteria(dir, folder_id, search_status, prestriction, folder_ids);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_search_criteria(dir, folder_id, search_status, prestriction, folder_ids);
	lpc_log(xbresult, dir, "get_search_criteria", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::set_search_criteria(const char *dir, cpid_t cpid, uint64_t folder_id, uint32_t search_flags, const RESTRICTION *prestriction, const EID_ARRAY *pfolder_ids, BOOL *b_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::set_search_criteria(dir, cpid, folder_id, search_flags, prestriction, pfolder_ids, b_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::set_search_criteria(dir, cpid, folder_id, search_flags, prestriction, pfolder_ids, b_result);
	lpc_log(xbresult, dir, "set_search_criteria", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::movecopy_message(const char *dir, cpid_t cpid, uint64_t message_id, uint64_t dst_fid, uint64_t dst_id, BOOL b_move, BOOL *b_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::movecopy_message(dir, cpid, message_id, dst_fid, dst_id, b_move, b_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::movecopy_message(dir, cpid, message_id, dst_fid, dst_id, b_move, b_result);
	lpc_log(xbresult, dir, "movecopy_message", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::movecopy_messages(const char *dir, cpid_t cpid, BOOL b_guest, const char *username, uint64_t src_fid, uint64_t dst_fid, BOOL b_copy, const EID_ARRAY *pmessage_ids, BOOL *b_partial)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::movecopy_messages(dir, cpid, b_guest, username, src_fid, dst_fid, b_copy, pmessage_ids, b_partial);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::movecopy_messages(dir, cpid, b_guest, username, src_fid, dst_fid, b_copy, pmessage_ids, b_partial);
	lpc_log(xbresult, dir, "movecopy_messages", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::movecopy_folder(const char *dir, cpid_t cpid, BOOL b_guest, const char *username, uint64_t src_pid, uint64_t src_fid, uint64_t dst_fid, const char *str_new, BOOL b_copy, ec_error_t *e_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::movecopy_folder(dir, cpid, b_guest, username, src_pid, src_fid, dst_fid, str_new, b_copy, e_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::movecopy_folder(dir, cpid, b_guest, username, src_pid, src_fid, dst_fid, str_new, b_copy, e_result);
	lpc_log(xbresult, dir, "movecopy_folder", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::delete_messages(const char *dir, cpid_t cpid, const char *username, uint64_t folder_id, const EID_ARRAY *pmessage_ids, BOOL b_hard, BOOL *b_partial)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::delete_messages(dir, cpid, username, folder_id, pmessage_ids, b_hard, b_partial);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::delete_messages(dir, cpid, username, folder_id, pmessage_ids, b_hard, b_partial);
	lpc_log(xbresult, dir, "delete_messages", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_message_brief(const char *dir, cpid_t cpid, uint64_t message_id, MESSAGE_CONTENT **pbrief)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_message_brief(dir, cpid, message_id, pbrief);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_message_brief(dir, cpid, message_id, pbrief);
	lpc_log(xbresult, dir, "get_message_brief", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::sum_hierarchy(const char *dir, uint64_t folder_id, const char *username, BOOL b_depth, uint32_t *count)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::sum_hierarchy(dir, folder_id, username, b_depth, count);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::sum_hierarchy(dir, folder_id, username, b_depth, count);
	lpc_log(xbresult, dir, "sum_hierarchy", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::load_hierarchy_table(const char *dir, uint64_t folder_id, const char *username, uint8_t table_flags, const RESTRICTION *prestriction, uint32_t *table_id, uint32_t *row_count)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::load_hierarchy_table(dir, folder_id, username, table_flags, prestriction, table_id, row_count);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::load_hierarchy_table(dir, folder_id, username, table_flags, prestriction, table_id, row_count);
	lpc_log(xbresult, dir, "load_hierarchy_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::sum_content(const char *dir, uint64_t folder_id, BOOL b_fai, BOOL b_deleted, uint32_t *count)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::sum_content(dir, folder_id, b_fai, b_deleted, count);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::sum_content(dir, folder_id, b_fai, b_deleted, count);
	lpc_log(xbresult, dir, "sum_content", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::load_content_table(const char *dir, cpid_t cpid, uint64_t folder_id, const char *username, uint8_t table_flags, const RESTRICTION *prestriction, const SORTORDER_SET *psorts, uint32_t *table_id, uint32_t *row_count)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::load_content_table(dir, cpid, folder_id, username, table_flags, prestriction, psorts, table_id, row_count);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::load_content_table(dir, cpid, folder_id, username, table_flags, prestriction, psorts, table_id, row_count);
	lpc_log(xbresult, dir, "load_content_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::reload_content_table(const char *dir, uint32_t table_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::reload_content_table(dir, table_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::reload_content_table(dir, table_id);
	lpc_log(xbresult, dir, "reload_content_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::load_permission_table(const char *dir, uint64_t folder_id, uint32_t table_flags, uint32_t *table_id, uint32_t *row_count)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::load_permission_table(dir, folder_id, table_flags, table_id, row_count);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::load_permission_table(dir, folder_id, table_flags, table_id, row_count);
	lpc_log(xbresult, dir, "load_permission_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::load_rule_table(const char *dir, uint64_t folder_id, uint8_t table_flags, const RESTRICTION *prestriction, uint32_t *table_id, uint32_t *row_count)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::load_rule_table(dir, folder_id, table_flags, prestriction, table_id, row_count);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::load_rule_table(dir, folder_id, table_flags, prestriction, table_id, row_count);
	lpc_log(xbresult, dir, "load_rule_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::unload_table(const char *dir, uint32_t table_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::unload_table(dir, table_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::unload_table(dir, table_id);
	lpc_log(xbresult, dir, "unload_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::sum_table(const char *dir, uint32_t table_id, uint32_t *rows)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::sum_table(dir, table_id, rows);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::sum_table(dir, table_id, rows);
	lpc_log(xbresult, dir, "sum_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::query_table(const char *dir, const char *username, cpid_t cpid, uint32_t table_id, proptag_cspan pproptags, uint32_t start_pos, int32_t row_needed, TARRAY_SET *set)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::query_table(dir, username, cpid, table_id, pproptags, start_pos, row_needed, set);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::query_table(dir, username, cpid, table_id, pproptags, start_pos, row_needed, set);
	lpc_log(xbresult, dir, "query_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::match_table(const char *dir, const char *username, cpid_t cpid, uint32_t table_id, BOOL b_forward, uint32_t start_pos, const RESTRICTION *pres, proptag_cspan pproptags, int32_t *position, TPROPVAL_ARRAY *propvals)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::match_table(dir, username, cpid, table_id, b_forward, start_pos, pres, pproptags, position, propvals);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::match_table(dir, username, cpid, table_id, b_forward, start_pos, pres, pproptags, position, propvals);
	lpc_log(xbresult, dir, "match_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::locate_table(const char *dir, uint32_t table_id, uint64_t inst_id, uint32_t inst_num, int32_t *position, uint32_t *row_type)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::locate_table(dir, table_id, inst_id, inst_num, position, row_type);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::locate_table(dir, table_id, inst_id, inst_num, position, row_type);
	lpc_log(xbresult, dir, "locate_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::read_table_row(const char *dir, const char *username, cpid_t cpid, uint32_t table_id, proptag_cspan pproptags, uint64_t inst_id, uint32_t inst_num, TPROPVAL_ARRAY *propvals)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::read_table_row(dir, username, cpid, table_id, pproptags, inst_id, inst_num, propvals);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::read_table_row(dir, username, cpid, table_id, pproptags, inst_id, inst_num, propvals);
	lpc_log(xbresult, dir, "read_table_row", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::mark_table(const char *dir, uint32_t table_id, uint32_t position, uint64_t *inst_id, uint32_t *inst_num, uint32_t *row_type)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::mark_table(dir, table_id, position, inst_id, inst_num, row_type);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::mark_table(dir, table_id, position, inst_id, inst_num, row_type);
	lpc_log(xbresult, dir, "mark_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_table_all_proptags(const char *dir, uint32_t table_id, PROPTAG_ARRAY *proptags)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_table_all_proptags(dir, table_id, proptags);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_table_all_proptags(dir, table_id, proptags);
	lpc_log(xbresult, dir, "get_table_all_proptags", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::expand_table(const char *dir, uint32_t table_id, uint64_t inst_id, BOOL *b_found, int32_t *position, uint32_t *row_count)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::expand_table(dir, table_id, inst_id, b_found, position, row_count);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::expand_table(dir, table_id, inst_id, b_found, position, row_count);
	lpc_log(xbresult, dir, "expand_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::collapse_table(const char *dir, uint32_t table_id, uint64_t inst_id, BOOL *b_found, int32_t *position, uint32_t *row_count)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::collapse_table(dir, table_id, inst_id, b_found, position, row_count);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::collapse_table(dir, table_id, inst_id, b_found, position, row_count);
	lpc_log(xbresult, dir, "collapse_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::store_table_state(const char *dir, uint32_t table_id, uint64_t inst_id, uint32_t inst_num, uint32_t *state_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::store_table_state(dir, table_id, inst_id, inst_num, state_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::store_table_state(dir, table_id, inst_id, inst_num, state_id);
	lpc_log(xbresult, dir, "store_table_state", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::restore_table_state(const char *dir, uint32_t table_id, uint32_t state_id, int32_t *position)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::restore_table_state(dir, table_id, state_id, position);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::restore_table_state(dir, table_id, state_id, position);
	lpc_log(xbresult, dir, "restore_table_state", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::is_msg_present(const char *dir, uint64_t folder_id, uint64_t message_id, BOOL *b_exist)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::is_msg_present(dir, folder_id, message_id, b_exist);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::is_msg_present(dir, folder_id, message_id, b_exist);
	lpc_log(xbresult, dir, "is_msg_present", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::is_msg_deleted(const char *dir, uint64_t message_id, BOOL *b_del)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::is_msg_deleted(dir, message_id, b_del);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::is_msg_deleted(dir, message_id, b_del);
	lpc_log(xbresult, dir, "is_msg_deleted", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::load_message_instance(const char *dir, const char *username, cpid_t cpid, BOOL b_new, uint64_t folder_id, uint64_t message_id, uint32_t *instance_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::load_message_instance(dir, username, cpid, b_new, folder_id, message_id, instance_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::load_message_instance(dir, username, cpid, b_new, folder_id, message_id, instance_id);
	lpc_log(xbresult, dir, "load_message_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::load_embedded_instance(const char *dir, BOOL b_new, uint32_t attachment_instance_id, uint32_t *instance_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::load_embedded_instance(dir, b_new, attachment_instance_id, instance_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::load_embedded_instance(dir, b_new, attachment_instance_id, instance_id);
	lpc_log(xbresult, dir, "load_embedded_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_embedded_cn(const char *dir, uint32_t instance_id, uint64_t **pcn)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_embedded_cn(dir, instance_id, pcn);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_embedded_cn(dir, instance_id, pcn);
	lpc_log(xbresult, dir, "get_embedded_cn", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::reload_message_instance(const char *dir, uint32_t instance_id, BOOL *b_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::reload_message_instance(dir, instance_id, b_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::reload_message_instance(dir, instance_id, b_result);
	lpc_log(xbresult, dir, "reload_message_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::clear_message_instance(const char *dir, uint32_t instance_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::clear_message_instance(dir, instance_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::clear_message_instance(dir, instance_id);
	lpc_log(xbresult, dir, "clear_message_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::read_message_instance(const char *dir, uint32_t instance_id, MESSAGE_CONTENT *msgctnt)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::read_message_instance(dir, instance_id, msgctnt);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::read_message_instance(dir, instance_id, msgctnt);
	lpc_log(xbresult, dir, "read_message_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::write_message_instance(const char *dir, uint32_t instance_id, const MESSAGE_CONTENT *pmsgctnt, BOOL b_force, PROPTAG_ARRAY *proptags, PROBLEM_ARRAY *problems)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::write_message_instance(dir, instance_id, pmsgctnt, b_force, proptags, problems);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::write_message_instance(dir, instance_id, pmsgctnt, b_force, proptags, problems);
	lpc_log(xbresult, dir, "write_message_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::load_attachment_instance(const char *dir, uint32_t message_instance_id, uint32_t attachment_num, uint32_t *instance_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::load_attachment_instance(dir, message_instance_id, attachment_num, instance_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::load_attachment_instance(dir, message_instance_id, attachment_num, instance_id);
	lpc_log(xbresult, dir, "load_attachment_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::create_attachment_instance(const char *dir, uint32_t message_instance_id, uint32_t *instance_id, uint32_t *attachment_num)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::create_attachment_instance(dir, message_instance_id, instance_id, attachment_num);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::create_attachment_instance(dir, message_instance_id, instance_id, attachment_num);
	lpc_log(xbresult, dir, "create_attachment_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::read_attachment_instance(const char *dir, uint32_t instance_id, ATTACHMENT_CONTENT *attctnt)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::read_attachment_instance(dir, instance_id, attctnt);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::read_attachment_instance(dir, instance_id, attctnt);
	lpc_log(xbresult, dir, "read_attachment_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::write_attachment_instance(const char *dir, uint32_t instance_id, const ATTACHMENT_CONTENT *pattctnt, BOOL b_force, PROBLEM_ARRAY *problems)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::write_attachment_instance(dir, instance_id, pattctnt, b_force, problems);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::write_attachment_instance(dir, instance_id, pattctnt, b_force, problems);
	lpc_log(xbresult, dir, "write_attachment_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::delete_message_instance_attachment(const char *dir, uint32_t message_instance_id, uint32_t attachment_num)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::delete_message_instance_attachment(dir, message_instance_id, attachment_num);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::delete_message_instance_attachment(dir, message_instance_id, attachment_num);
	lpc_log(xbresult, dir, "delete_message_instance_attachment", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::flush_instance(const char *dir, uint32_t instance_id, ec_error_t *e_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::flush_instance(dir, instance_id, e_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::flush_instance(dir, instance_id, e_result);
	lpc_log(xbresult, dir, "flush_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::unload_instance(const char *dir, uint32_t instance_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::unload_instance(dir, instance_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::unload_instance(dir, instance_id);
	lpc_log(xbresult, dir, "unload_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_instance_all_proptags(const char *dir, uint32_t instance_id, PROPTAG_ARRAY *proptags)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_instance_all_proptags(dir, instance_id, proptags);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_instance_all_proptags(dir, instance_id, proptags);
	lpc_log(xbresult, dir, "get_instance_all_proptags", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_instance_properties(const char *dir, uint32_t size_limit, uint32_t instance_id, proptag_cspan pproptags, TPROPVAL_ARRAY *propvals)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_instance_properties(dir, size_limit, instance_id, pproptags, propvals);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_instance_properties(dir, size_limit, instance_id, pproptags, propvals);
	lpc_log(xbresult, dir, "get_instance_properties", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::set_instance_properties(const char *dir, uint32_t instance_id, const TPROPVAL_ARRAY *pproperties, PROBLEM_ARRAY *problems)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::set_instance_properties(dir, instance_id, pproperties, problems);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::set_instance_properties(dir, instance_id, pproperties, problems);
	lpc_log(xbresult, dir, "set_instance_properties", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::remove_instance_properties(const char *dir, uint32_t instance_id, proptag_cspan pproptags, PROBLEM_ARRAY *problems)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::remove_instance_properties(dir, instance_id, pproptags, problems);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::remove_instance_properties(dir, instance_id, pproptags, problems);
	lpc_log(xbresult, dir, "remove_instance_properties", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::is_descendant_instance(const char *dir, uint32_t parent_iid, uint32_t child_iid, BOOL *b_included)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::is_descendant_instance(dir, parent_iid, child_iid, b_included);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::is_descendant_instance(dir, parent_iid, child_iid, b_included);
	lpc_log(xbresult, dir, "is_descendant_instance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::empty_message_instance_rcpts(const char *dir, uint32_t instance_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::empty_message_instance_rcpts(dir, instance_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::empty_message_instance_rcpts(dir, instance_id);
	lpc_log(xbresult, dir, "empty_message_instance_rcpts", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_message_instance_rcpts_num(const char *dir, uint32_t instance_id, uint16_t *num)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_message_instance_rcpts_num(dir, instance_id, num);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_message_instance_rcpts_num(dir, instance_id, num);
	lpc_log(xbresult, dir, "get_message_instance_rcpts_num", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_message_instance_rcpts_all_proptags(const char *dir, uint32_t instance_id, PROPTAG_ARRAY *proptags)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_message_instance_rcpts_all_proptags(dir, instance_id, proptags);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_message_instance_rcpts_all_proptags(dir, instance_id, proptags);
	lpc_log(xbresult, dir, "get_message_instance_rcpts_all_proptags", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_message_instance_rcpts(const char *dir, uint32_t instance_id, uint32_t row_id, uint16_t need_count, TARRAY_SET *set)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_message_instance_rcpts(dir, instance_id, row_id, need_count, set);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_message_instance_rcpts(dir, instance_id, row_id, need_count, set);
	lpc_log(xbresult, dir, "get_message_instance_rcpts", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::update_message_instance_rcpts(const char *dir, uint32_t instance_id, const TARRAY_SET *pset)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::update_message_instance_rcpts(dir, instance_id, pset);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::update_message_instance_rcpts(dir, instance_id, pset);
	lpc_log(xbresult, dir, "update_message_instance_rcpts", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::copy_instance_rcpts(const char *dir, BOOL b_force, uint32_t src_instance_id, uint32_t dst_instance_id, BOOL *b_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::copy_instance_rcpts(dir, b_force, src_instance_id, dst_instance_id, b_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::copy_instance_rcpts(dir, b_force, src_instance_id, dst_instance_id, b_result);
	lpc_log(xbresult, dir, "copy_instance_rcpts", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::empty_message_instance_attachments(const char *dir, uint32_t instance_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::empty_message_instance_attachments(dir, instance_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::empty_message_instance_attachments(dir, instance_id);
	lpc_log(xbresult, dir, "empty_message_instance_attachments", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_message_instance_attachments_num(const char *dir, uint32_t instance_id, uint16_t *num)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_message_instance_attachments_num(dir, instance_id, num);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_message_instance_attachments_num(dir, instance_id, num);
	lpc_log(xbresult, dir, "get_message_instance_attachments_num", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_message_instance_attachment_table_all_proptags(const char *dir, uint32_t instance_id, PROPTAG_ARRAY *proptags)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_message_instance_attachment_table_all_proptags(dir, instance_id, proptags);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_message_instance_attachment_table_all_proptags(dir, instance_id, proptags);
	lpc_log(xbresult, dir, "get_message_instance_attachment_table_all_proptags", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::query_message_instance_attachment_table(const char *dir, uint32_t instance_id, proptag_cspan pproptags, uint32_t start_pos, int32_t row_needed, TARRAY_SET *set)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::query_message_instance_attachment_table(dir, instance_id, pproptags, start_pos, row_needed, set);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::query_message_instance_attachment_table(dir, instance_id, pproptags, start_pos, row_needed, set);
	lpc_log(xbresult, dir, "query_message_instance_attachment_table", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::copy_instance_attachments(const char *dir, BOOL b_force, uint32_t src_instance_id, uint32_t dst_instance_id, BOOL *b_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::copy_instance_attachments(dir, b_force, src_instance_id, dst_instance_id, b_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::copy_instance_attachments(dir, b_force, src_instance_id, dst_instance_id, b_result);
	lpc_log(xbresult, dir, "copy_instance_attachments", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::set_message_instance_conflict(const char *dir, uint32_t instance_id, const MESSAGE_CONTENT *pmsgctnt)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::set_message_instance_conflict(dir, instance_id, pmsgctnt);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::set_message_instance_conflict(dir, instance_id, pmsgctnt);
	lpc_log(xbresult, dir, "set_message_instance_conflict", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_message_rcpts(const char *dir, uint64_t message_id, TARRAY_SET *set)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_message_rcpts(dir, message_id, set);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_message_rcpts(dir, message_id, set);
	lpc_log(xbresult, dir, "get_message_rcpts", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_message_properties(const char *dir, const char *username, cpid_t cpid, uint64_t message_id, proptag_cspan pproptags, TPROPVAL_ARRAY *propvals)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_message_properties(dir, username, cpid, message_id, pproptags, propvals);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_message_properties(dir, username, cpid, message_id, pproptags, propvals);
	lpc_log(xbresult, dir, "get_message_properties", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::set_message_properties(const char *dir, const char *username, cpid_t cpid, uint64_t message_id, const TPROPVAL_ARRAY *pproperties, PROBLEM_ARRAY *problems)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::set_message_properties(dir, username, cpid, message_id, pproperties, problems);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::set_message_properties(dir, username, cpid, message_id, pproperties, problems);
	lpc_log(xbresult, dir, "set_message_properties", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::set_message_read_state(const char *dir, const char *username, uint64_t message_id, uint8_t mark_as_read, uint64_t *read_cn)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::set_message_read_state(dir, username, message_id, mark_as_read, read_cn);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::set_message_read_state(dir, username, message_id, mark_as_read, read_cn);
	lpc_log(xbresult, dir, "set_message_read_state", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::remove_message_properties(const char *dir, cpid_t cpid, uint64_t message_id, proptag_cspan pproptags)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::remove_message_properties(dir, cpid, message_id, pproptags);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::remove_message_properties(dir, cpid, message_id, pproptags);
	lpc_log(xbresult, dir, "remove_message_properties", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::allocate_message_id(const char *dir, uint64_t folder_id, uint64_t *message_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::allocate_message_id(dir, folder_id, message_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::allocate_message_id(dir, folder_id, message_id);
	lpc_log(xbresult, dir, "allocate_message_id", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::allocate_cn(const char *dir, uint64_t *cn)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::allocate_cn(dir, cn);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::allocate_cn(dir, cn);
	lpc_log(xbresult, dir, "allocate_cn", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::mark_modified(const char *dir, uint64_t message_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::mark_modified(dir, message_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::mark_modified(dir, message_id);
	lpc_log(xbresult, dir, "mark_modified", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::try_mark_submit(const char *dir, uint64_t message_id, BOOL *b_marked)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::try_mark_submit(dir, message_id, b_marked);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::try_mark_submit(dir, message_id, b_marked);
	lpc_log(xbresult, dir, "try_mark_submit", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::clear_submit(const char *dir, uint64_t message_id, BOOL b_unsent)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::clear_submit(dir, message_id, b_unsent);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::clear_submit(dir, message_id, b_unsent);
	lpc_log(xbresult, dir, "clear_submit", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::link_message(const char *dir, cpid_t cpid, uint64_t folder_id, uint64_t message_id, BOOL *b_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::link_message(dir, cpid, folder_id, message_id, b_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::link_message(dir, cpid, folder_id, message_id, b_result);
	lpc_log(xbresult, dir, "link_message", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::unlink_message(const char *dir, cpid_t cpid, uint64_t folder_id, uint64_t message_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::unlink_message(dir, cpid, folder_id, message_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::unlink_message(dir, cpid, folder_id, message_id);
	lpc_log(xbresult, dir, "unlink_message", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::rule_new_message(const char *dir, const char *username, cpid_t cpid, uint64_t folder_id, uint64_t message_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::rule_new_message(dir, username, cpid, folder_id, message_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::rule_new_message(dir, username, cpid, folder_id, message_id);
	lpc_log(xbresult, dir, "rule_new_message", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::set_message_timer(const char *dir, uint64_t message_id, uint32_t timer_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::set_message_timer(dir, message_id, timer_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::set_message_timer(dir, message_id, timer_id);
	lpc_log(xbresult, dir, "set_message_timer", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_message_timer(const char *dir, uint64_t message_id, uint32_t **ptimer_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_message_timer(dir, message_id, ptimer_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_message_timer(dir, message_id, ptimer_id);
	lpc_log(xbresult, dir, "get_message_timer", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::empty_folder_permission(const char *dir, uint64_t folder_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::empty_folder_permission(dir, folder_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::empty_folder_permission(dir, folder_id);
	lpc_log(xbresult, dir, "empty_folder_permission", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::update_folder_permission(const char *dir, uint64_t folder_id, BOOL b_freebusy, uint16_t count, const PERMISSION_DATA *prow)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::update_folder_permission(dir, folder_id, b_freebusy, count, prow);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::update_folder_permission(dir, folder_id, b_freebusy, count, prow);
	lpc_log(xbresult, dir, "update_folder_permission", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::empty_folder_rule(const char *dir, uint64_t folder_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::empty_folder_rule(dir, folder_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::empty_folder_rule(dir, folder_id);
	lpc_log(xbresult, dir, "empty_folder_rule", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::update_folder_rule(const char *dir, uint64_t folder_id, uint16_t count, const RULE_DATA *prow, BOOL *b_exceed)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::update_folder_rule(dir, folder_id, count, prow, b_exceed);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::update_folder_rule(dir, folder_id, count, prow, b_exceed);
	lpc_log(xbresult, dir, "update_folder_rule", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::deliver_message(const char *dir, const char *from_address, const char *account, cpid_t cpid, uint32_t dlflags, const MESSAGE_CONTENT *pmsg, const char *pdigest, uint64_t *folder_id, uint64_t *message_id, uint32_t *result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::deliver_message(dir, from_address, account, cpid, dlflags, pmsg, pdigest, folder_id, message_id, result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::deliver_message(dir, from_address, account, cpid, dlflags, pmsg, pdigest, folder_id, message_id, result);
	lpc_log(xbresult, dir, "deliver_message", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::write_message(const char *dir, cpid_t cpid, uint64_t folder_id, const MESSAGE_CONTENT *pmsgctnt, const std::string &digest, uint64_t *outmid, uint64_t *outcn, ec_error_t *e_result)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::write_message(dir, cpid, folder_id, pmsgctnt, digest, outmid, outcn, e_result);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::write_message(dir, cpid, folder_id, pmsgctnt, digest, outmid, outcn, e_result);
	lpc_log(xbresult, dir, "write_message", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::read_message(const char *dir, const char *username, cpid_t cpid, uint64_t message_id, MESSAGE_CONTENT **pmsgctnt)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::read_message(dir, username, cpid, message_id, pmsgctnt);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::read_message(dir, username, cpid, message_id, pmsgctnt);
	lpc_log(xbresult, dir, "read_message", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_content_sync(const char *dir, uint64_t folder_id, const char *username, const gromox::idset *pgiven, const gromox::idset *pseen, const gromox::idset *pseen_fai, const gromox::idset *pread, cpid_t cpid, const RESTRICTION *prestriction, BOOL b_ordered, uint32_t *fai_count, uint64_t *fai_total, uint32_t *normal_count, uint64_t *normal_total, EID_ARRAY *updated_mids, EID_ARRAY *chg_mids, uint64_t *last_cn, EID_ARRAY *given_mids, EID_ARRAY *deleted_mids, EID_ARRAY *nolonger_mids, EID_ARRAY *read_mids, EID_ARRAY *unread_mids, uint64_t *last_readcn)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_content_sync(dir, folder_id, username, pgiven, pseen, pseen_fai, pread, cpid, prestriction, b_ordered, fai_count, fai_total, normal_count, normal_total, updated_mids, chg_mids, last_cn, given_mids, deleted_mids, nolonger_mids, read_mids, unread_mids, last_readcn);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_content_sync(dir, folder_id, username, pgiven, pseen, pseen_fai, pread, cpid, prestriction, b_ordered, fai_count, fai_total, normal_count, normal_total, updated_mids, chg_mids, last_cn, given_mids, deleted_mids, nolonger_mids, read_mids, unread_mids, last_readcn);
	lpc_log(xbresult, dir, "get_content_sync", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_hierarchy_sync(const char *dir, uint64_t folder_id, const char *username, const gromox::idset *pgiven, const gromox::idset *pseen, FOLDER_CHANGES *fldchgs, uint64_t *last_cn, EID_ARRAY *given_fids, EID_ARRAY *deleted_fids)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_hierarchy_sync(dir, folder_id, username, pgiven, pseen, fldchgs, last_cn, given_fids, deleted_fids);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_hierarchy_sync(dir, folder_id, username, pgiven, pseen, fldchgs, last_cn, given_fids, deleted_fids);
	lpc_log(xbresult, dir, "get_hierarchy_sync", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::allocate_ids(const char *dir, uint32_t count, uint64_t *begin_eid)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::allocate_ids(dir, count, begin_eid);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::allocate_ids(dir, count, begin_eid);
	lpc_log(xbresult, dir, "allocate_ids", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::subscribe_notification(const char *dir, uint16_t notification_type, BOOL b_whole, uint64_t folder_id, uint64_t message_id, uint32_t *sub_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::subscribe_notification(dir, notification_type, b_whole, folder_id, message_id, sub_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::subscribe_notification(dir, notification_type, b_whole, folder_id, message_id, sub_id);
	lpc_log(xbresult, dir, "subscribe_notification", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::unsubscribe_notification(const char *dir, uint32_t sub_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::unsubscribe_notification(dir, sub_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::unsubscribe_notification(dir, sub_id);
	lpc_log(xbresult, dir, "unsubscribe_notification", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::transport_new_mail(const char *dir, uint64_t folder_id, uint64_t message_id, uint32_t message_flags, const char *pstr_class)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::transport_new_mail(dir, folder_id, message_id, message_flags, pstr_class);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::transport_new_mail(dir, folder_id, message_id, message_flags, pstr_class);
	lpc_log(xbresult, dir, "transport_new_mail", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::check_contact_address(const char *dir, const char *paddress, BOOL *b_found)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::check_contact_address(dir, paddress, b_found);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::check_contact_address(dir, paddress, b_found);
	lpc_log(xbresult, dir, "check_contact_address", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::get_public_folder_unread_count(const char *dir, const char *username, uint64_t folder_id, uint32_t *count)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::get_public_folder_unread_count(dir, username, folder_id, count);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::get_public_folder_unread_count(dir, username, folder_id, count);
	lpc_log(xbresult, dir, "get_public_folder_unread_count", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::vacuum(const char *dir)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::vacuum(dir);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::vacuum(dir);
	lpc_log(xbresult, dir, "vacuum", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::unload_store(const char *dir)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::unload_store(dir);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::unload_store(dir);
	lpc_log(xbresult, dir, "unload_store", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::notify_new_mail(const char *dir, uint64_t folder_id, uint64_t message_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::notify_new_mail(dir, folder_id, message_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::notify_new_mail(dir, folder_id, message_id);
	lpc_log(xbresult, dir, "notify_new_mail", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::store_eid_to_user(const char *dir, const STORE_ENTRYID *store_eid, char **maildir, unsigned int *user_id, unsigned int *domain_id)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::store_eid_to_user(dir, store_eid, maildir, user_id, domain_id);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::store_eid_to_user(dir, store_eid, maildir, user_id, domain_id);
	lpc_log(xbresult, dir, "store_eid_to_user", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::purge_softdelete(const char *dir, const char *username, uint64_t folder_id, uint32_t del_flags, gromox::mapitime_t cutoff, uint32_t *cnt_folders, uint32_t *cnt_messages, uint64_t *sz_normal, uint64_t *sz_fai)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::purge_softdelete(dir, username, folder_id, del_flags, cutoff, cnt_folders, cnt_messages, sz_normal, sz_fai);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::purge_softdelete(dir, username, folder_id, del_flags, cutoff, cnt_folders, cnt_messages, sz_normal, sz_fai);
	lpc_log(xbresult, dir, "purge_softdelete", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::purge_datafiles(const char *dir)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::purge_datafiles(dir);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::purge_datafiles(dir);
	lpc_log(xbresult, dir, "purge_datafiles", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::autoreply_tsquery(const char *dir, const char *peer, uint64_t window, uint64_t *tdiff)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::autoreply_tsquery(dir, peer, window, tdiff);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::autoreply_tsquery(dir, peer, window, tdiff);
	lpc_log(xbresult, dir, "autoreply_tsquery", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::autoreply_tsupdate(const char *dir, const char *peer)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::autoreply_tsupdate(dir, peer);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::autoreply_tsupdate(dir, peer);
	lpc_log(xbresult, dir, "autoreply_tsupdate", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::recalc_store_size(const char *dir, uint32_t flags)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::recalc_store_size(dir, flags);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::recalc_store_size(dir, flags);
	lpc_log(xbresult, dir, "recalc_store_size", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::imapfile_read(const char *dir, const std::string &type, const std::string &mid, std::string *data)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::imapfile_read(dir, type, mid, data);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::imapfile_read(dir, type, mid, data);
	lpc_log(xbresult, dir, "imapfile_read", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::imapfile_write(const char *dir, const std::string &type, const std::string &mid, const std::string &data)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::imapfile_write(dir, type, mid, data);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::imapfile_write(dir, type, mid, data);
	lpc_log(xbresult, dir, "imapfile_write", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::imapfile_delete(const char *dir, const std::string &type, const std::string &mid)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::imapfile_delete(dir, type, mid);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::imapfile_delete(dir, type, mid);
	lpc_log(xbresult, dir, "imapfile_delete", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::cgkreset(const char *dir, uint32_t flags)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::cgkreset(dir, flags);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::cgkreset(dir, flags);
	lpc_log(xbresult, dir, "cgkreset", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::set_maintenance(const char *dir, uint32_t mode)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::set_maintenance(dir, mode);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::set_maintenance(dir, mode);
	lpc_log(xbresult, dir, "set_maintenance", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::autoreply_getprop(const char *dir, cpid_t cpid, proptag_cspan pproptags, TPROPVAL_ARRAY *propvals)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::autoreply_getprop(dir, cpid, pproptags, propvals);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::autoreply_getprop(dir, cpid, pproptags, propvals);
	lpc_log(xbresult, dir, "autoreply_getprop", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::autoreply_setprop(const char *dir, cpid_t cpid, const TPROPVAL_ARRAY *ppropvals, PROBLEM_ARRAY *problems)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::autoreply_setprop(dir, cpid, ppropvals, problems);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::autoreply_setprop(dir, cpid, ppropvals, problems);
	lpc_log(xbresult, dir, "autoreply_setprop", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::read_delegates(const char *dir, uint32_t mode, std::vector<std::string> *userlist)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::read_delegates(dir, mode, userlist);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::read_delegates(dir, mode, userlist);
	lpc_log(xbresult, dir, "read_delegates", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

BOOL exmdb_client_local::write_delegates(const char *dir, uint32_t mode, const std::vector<std::string> &userlist)
{
	BOOL xb_private;

	if (!exmdb_client_can_use_lpc(dir, &xb_private))
		return exmdb_client_remote::write_delegates(dir, mode, userlist);
	auto tstart = gromox::tp_now();
	exmdb_server::build_env(EM_LOCAL | (xb_private ? EM_PRIVATE : 0), dir);
	auto xbresult = exmdb_server::write_delegates(dir, mode, userlist);
	lpc_log(xbresult, dir, "write_delegates", tstart, gromox::tp_now());
	exmdb_server::free_env();
	return xbresult;
}

