/*=============================================================================
ttyrpld - TTY replay daemon
kernel/rpldev.c - Kernel interface for `ttyrpld`
  Copyright (C) Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)
  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <asm/semaphore.h>
#include <asm/uaccess.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/rpl.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/wait.h>

#include <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
#    define KERNEL_2_4 1
#    define P_MAGIC MAGIC_2_4
#else
#    define P_MAGIC MAGIC_2_6
#endif

#include "../include/rpl_ioctl.h"
#include "../include/rpl_packet.h"
#ifndef KBUILD_BASENAME
#    define KBUILD_BASENAME "rpldev"
#endif
#define __STRINGIFY_EXP(__str) #__str
#define __STRINGIFY(__str) __STRINGIFY_EXP(__str)
#define PREFIX __STRINGIFY(KBUILD_BASENAME) ": "

// Stage 2 functions
static int krn_open(struct tty_struct *, struct tty_struct *);
static int krn_read(const char *, size_t, struct tty_struct *);
static int krn_write(const char *, size_t, struct tty_struct *);
static int krn_close(struct tty_struct *);
static int krn_ioctl(struct tty_struct *, struct tty_struct *, unsigned int,
 unsigned long);

// Stage 3 functions
static int uif_open(struct inode *, struct file *);
static ssize_t uif_read(struct file *, char *, size_t, loff_t *);
static loff_t uif_llseek(struct file *, loff_t, int);
static int uif_ioctl(struct inode *, struct file *, unsigned int,
 unsigned long);
static unsigned int uif_poll(struct file *, poll_table *);
static int uif_close(struct inode *, struct file *);

// Local functions
inline static ssize_t avail_R(void);
inline static ssize_t avail_W(void);
inline static int imin(int, int);
inline static void get_tty_devnr(uint32_t *, struct tty_struct *);
static int mv_buffer(struct rpld_packet *, const char *, size_t);
inline static void mv_buffer2(const void *, size_t);
inline static void mv_user(void *, size_t);

// Variables
static DECLARE_WAIT_QUEUE_HEAD(GV_inq);
static DECLARE_MUTEX(GV_open_lock);
static DECLARE_MUTEX(GV_buf_lock);
static char *GV_buffer = NULL, *GV_bufr = NULL, *GV_bufw = NULL;
static size_t GI_bufsize = 16384; // TTY driver has 4096
static struct file_operations GI_fops = {
    open:    uif_open,
    read:    uif_read,
    llseek:  uif_llseek,
    ioctl:   uif_ioctl,
    poll:    uif_poll,
    release: uif_close,
    owner:   THIS_MODULE,
};
static struct miscdevice GI_misc =
 {minor: MISC_DYNAMIC_MINOR, name: "rpl", fops: &GI_fops};

// Module info
int __init init_module(void);
void __exit cleanup_module(void);
MODULE_DESCRIPTION("ttyrpld Kernel interface");
MODULE_AUTHOR("Jan Engelhardt <jengelh [at] linux01 gwdg de>");
MODULE_LICENSE("GPL");
MODULE_PARM(GI_bufsize, "i");
MODULE_PARM_DESC(GI_bufsize, "Buffer size (default 16K)");
#ifdef KERNEL_2_4
EXPORT_NO_SYMBOLS;
#endif

//-----------------------------------------------------------------------------
int __init init_module(void) {
    int eax;

    /* Devfs was a nice invention, but since it was obsoleted, I am currently
    missing the "mknod-on-insmod" functionality. However, misc_register() is
    that smart to create a node in /dev (using regular fs functions, though),
    emulating devfs behavior. */
    if((eax = misc_register(&GI_misc)) != 0) { cleanup_module(); return eax; }
    return 0;
}

void __exit cleanup_module(void) {
    misc_deregister(&GI_misc);
    return;
}

//-----------------------------------------------------------------------------
static int krn_open(struct tty_struct *tty, struct tty_struct *cur_tty) {
    struct rpld_packet p;
    get_tty_devnr(&p.dev, tty);
    p.event = EV_OPEN;
    p.magic = P_MAGIC;
    p.size  = 0;
    return mv_buffer(&p, NULL, 0);
}

static int krn_read(const char *buf, size_t count, struct tty_struct *tty) {
    /* The data flow is a bit weird at first. krn_read() gets the data on its
    way between ttyDriver(master) -> /dev/stdin(slave), meaning this function
    is called when you hit the keyboard. */
    struct rpld_packet p;
    get_tty_devnr(&p.dev, tty);
    p.event = EV_READ;
    p.magic = P_MAGIC;
    p.size  = count;
    return mv_buffer(&p, buf, count);
}

static int krn_write(const char *buf, size_t count, struct tty_struct *tty) {
    // Data flow: /dev/stdout(slave) -> tty driver(master)
    struct rpld_packet p;
    get_tty_devnr(&p.dev, tty);
    p.event = EV_WRITE;
    p.magic = P_MAGIC;
    p.size  = count;
    return mv_buffer(&p, buf, count);
}

static int krn_ioctl(struct tty_struct *tty, struct tty_struct *real_tty,
 unsigned int cmd, unsigned long arg) {
    char buf[sizeof(cmd) + sizeof(arg)];
    struct rpld_packet p;

    get_tty_devnr(&p.dev, tty);
    p.event = EV_IOCTL;
    p.magic = P_MAGIC;
    p.size  = sizeof(buf);

    memcpy(buf, &cmd, sizeof(cmd));
    memcpy(buf + sizeof(cmd), &arg, sizeof(arg));
    return mv_buffer(&p, buf, sizeof(buf));
}

static int krn_close(struct tty_struct *tty) {
    struct rpld_packet p;
    get_tty_devnr(&p.dev, tty);
    p.event = EV_CLOSE;
    p.magic = P_MAGIC;
    p.size  = 0;
    return mv_buffer(&p, NULL, 0);
}

//-----------------------------------------------------------------------------
static int uif_open(struct inode *inode, struct file *filp) {
    // This one is called when the driver /dev/misc/rpl has been opened.
    if(inode != NULL) {
        inode->i_mtime = CURRENT_TIME;
        inode->i_mode &= ~(S_IWUGO | S_IXUGO);
    }

    /* The RPL device is read-only. Absolutely. No single O_WRONLY or O_RDWR
    is allowed. */
    if(filp->f_mode & 0x2) { return -EPERM; }

    /* The RPL device can only be opened once, since otherwise, different
    packets could go to different readers. */
    if(down_trylock(&GV_open_lock) != 0) { return -EBUSY; }

    if((GV_buffer = kmalloc(GI_bufsize, GFP_USER)) == NULL) { return -ENOMEM; }
    GV_bufr = GV_bufw = GV_buffer;

    /* Update links. I do it here 'cause I do not want memory copying (from the
    tty driver to the buffer) when there is no one to read. */
    rpl_qopen  = krn_open;
    rpl_qread  = krn_read;
    rpl_qwrite = krn_write;
    rpl_qclose = krn_close;
    rpl_qioctl = krn_ioctl;

    printk(KERN_DEBUG PREFIX "RPL device opened\n");

    /* The inode's times are changed as follows:
            Access Time: if data is read from the device
      Inode Change Time: when the device is successfully opened
      Modification Time: whenever the device is opened
    */
    if(inode != NULL) { inode->i_ctime = CURRENT_TIME; }
    return 0;
}

static ssize_t uif_read(struct file *filp, char *buf, size_t count,
 loff_t *ppos) {
    /* No seeking allowed. No positional reading allowed either. For skipping
    data, use SEEK_CUR with lseek(). */
    if(ppos != &filp->f_pos) { return -ESPIPE; }

    // Nothing read, nothing done
    if(count == 0) { return 0; }

    // Must sleep as long as there is no data.
    if(down_interruptible(&GV_buf_lock)) { return -ERESTARTSYS; }
    while(GV_bufr == GV_bufw) {
        up(&GV_buf_lock);
        if(filp->f_flags & O_NONBLOCK) { return -EAGAIN; }
        if(wait_event_interruptible(GV_inq, (GV_bufr != GV_bufw))) {
            return -ERESTARTSYS;
        }
        if(down_interruptible(&GV_buf_lock)) { return -ERESTARTSYS; }
    }

    // Data is available, so give it to the user
    count = imin(count, avail_R());
    mv_user(buf, count);

    up(&GV_buf_lock);
    filp->f_dentry->d_inode->i_atime = CURRENT_TIME;
    return count;
}

static loff_t uif_llseek(struct file *filp, loff_t offset, int origin) {
    /* Since you can not seek in the circular buffer, the only purpose of
    lseek() is to skip some bytes and "manually" advance the read pointer RP
    (GV_bufr). Thus, the only accepted origin is SEEK_CUR (1). */

    if(origin != 1) { return -ESPIPE; }
    if(down_interruptible(&GV_buf_lock)) { return -ERESTARTSYS; }

    GV_bufr += imin(offset, avail_R());
    if(GV_bufr > GV_buffer + GI_bufsize) { GV_bufr -= GI_bufsize; }

    up(&GV_buf_lock);
    return 0;
}

static int uif_ioctl(struct inode *inode, struct file *filp, unsigned int cmd,
 unsigned long arg) {
    if(_IOC_TYPE(cmd) != RPL_IOC_MAGIC) { return -ENOTTY; }
    if(_IOC_NR(cmd) >= RPL_IOC__MAXNUM) { return -ENOTTY; }

    switch(cmd) {
        case RPL_IOC_GETBUFSIZE:
            return GI_bufsize;
    }
    return 0;
}

static unsigned int uif_poll(struct file *filp, poll_table *wait) {
    poll_wait(filp, &GV_inq, wait);
    return POLLIN | POLLRDNORM;
}

static int uif_close(struct inode *inode, struct file *filp) {
    printk(KERN_DEBUG PREFIX "RPL device closed\n");
    rpl_qopen  = NULL;
    rpl_qread  = NULL;
    rpl_qwrite = NULL;
    rpl_qioctl = NULL;
    rpl_qclose = NULL;
    kfree(GV_buffer);
    up(&GV_open_lock);
    return 0;
}

//-----------------------------------------------------------------------------
inline static ssize_t avail_R(void) {
    // Return the number of available bytes to read
    if(GV_bufw >= GV_bufr) { return GV_bufw - GV_bufr; }
    return GV_bufw + GI_bufsize - GV_bufr;
}

inline static ssize_t avail_W(void) {
    // Return the number of available bytes to write
    if(GV_bufw >= GV_bufr) { return GV_bufr + GI_bufsize - GV_bufw - 1; }
    return GV_bufr - GV_bufw - 1;
}

inline static void get_tty_devnr(uint32_t *dev, struct tty_struct *tty) {
#ifdef KERNEL_2_4
    *dev = tty->device;
#else
    *dev = MKDEV(tty->driver->major, tty->driver->minor_start + tty->index);
#endif
    return;
}

inline static int imin(int a, int b) {
    return a < b ? a : b;
}

static int mv_buffer(struct rpld_packet *p, const char *buf, size_t count) {
    /* Copy the contents the tty driver received to our circulary buffer and
    also add a header (rpld_packet) to it so that the userspace daemon can
    recognize the type. */
    static int GI_overrun = 0;

    down_interruptible(&GV_buf_lock);

    if(avail_W() < sizeof(struct rpld_packet) + count) {
        if(!GI_overrun) {
            printk(KERN_INFO PREFIX "mv_buffer(): Buffer overrun\n");
            ++GI_overrun;
        }
        up(&GV_buf_lock);
        return -ENOSPC;
    } else {
        GI_overrun = 0;
    }

    mv_buffer2(p, sizeof(struct rpld_packet));
    if(count > 0) { mv_buffer2(buf, count); }

    up(&GV_buf_lock);
    wake_up(&GV_inq);
    return count;
}

inline static void mv_buffer2(const void *src, size_t count) {
    /* This function is responsible for copying (a specific amount of
    arbitrary data) into the circulary buffer. (Taking the wrappage into
    account!) The parent function -- mv_buffer() in this case -- must make sure
    there is enough room. */

    size_t x = GV_buffer + GI_bufsize - GV_bufw;
    if(count <= x) {
        memcpy(GV_bufw, src, count);
        GV_bufw += count;
    } else {
        memcpy(GV_bufw, src, x);
        memcpy(GV_buffer, src + x, count - x);
        GV_bufw = GV_buffer + count - x;
    }
    return;
}

inline static void mv_user(void *dest, size_t count) {
    /* mv_user() reads <count> bytes from the circulary buffer and puts it into
    userspace memory. */
    int x = GV_buffer + GI_bufsize - GV_bufr;

    /* Since this operation may block due to userspace page faults, the
    userspace daemon should really have <dest> memory-locked. */
    if(count <= x) {
        copy_to_user(dest, GV_bufr, count);
        GV_bufr += count;
    } else {
        copy_to_user(dest, GV_bufr, x);
        copy_to_user(dest + x, GV_buffer, count - x);
        GV_bufr = GV_buffer + count - x;
    }

    return;
}

//==[ End of file ]============================================================
