/*=============================================================================
ttyrpld - TTY replay daemon
user/replay.c - Realtime TTY log analyzer
  Copyright (C) Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)
  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/stat.h>
#include <sys/time.h> // gettimeofday()
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <popt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h> // nanosleep()
#include <unistd.h>

#include "dev.h"
#include "rpl_packet.h"
#include "ushared.h"

#define MICROSECOND 1000000
#define NANOSECOND 1000000000
#define BUFSIZE 4096

struct {
    int debug;
    double factor;
    unsigned long ovcorr;
} opt = {
    debug: 0,
    factor: 1.0,
    ovcorr: 0,
};

static void replay_file(int);

static unsigned long calc_ovcorr(unsigned long, int) __attribute__((unused));
static int get_options(int *, const char ***);
inline static int imin(int, int);
static int nsleep_std(struct timespec *, unsigned long *);
static int nsleep_ovcorr(struct timespec *, unsigned long *);
inline static void tv_delta_to_nano(const struct timeval *,
 const struct timeval *, struct timespec *);

static int (*nsleep_chosen)(struct timespec *, unsigned long *) = nsleep_std;

//-----------------------------------------------------------------------------
int main(int argc, char **argv) {
    if(!get_options(&argc, (const char ***)&argv)) { return EXIT_FAILURE; }

    while(*argv != NULL) {
        int fd;
        if((fd = open(*argv, O_RDONLY)) < 0) {
            fprintf(stderr, "Could not open %s: %s\n",
             *argv, strerror(errno));
            ++argv;
            continue;
        }

        replay_file(fd);
        close(fd);
        ++argv;
    }

    return EXIT_SUCCESS;
}

static void replay_file(int fd) {
    char buf[BUFSIZE];
    struct log_packet packet;
    struct timeval stamp;
    ssize_t eax = read(fd, &packet, sizeof(struct log_packet));
    unsigned long xdelay = 0;

    if(eax == (ssize_t)sizeof(struct log_packet)) while(1) {
        size_t have_read, read_left;

        memcpy(&stamp, &packet.tv, sizeof(struct timeval));
        read_left = packet.p.size;

        while(read_left > 0) {
            have_read = read(fd, buf, imin(read_left, BUFSIZE));
            if(packet.p.event == EV_WRITE) {
                fwrite(buf, have_read, 1, stdout);
            }
            read_left -= have_read;
        }

        fflush(stdout);
        if(read(fd, &packet, sizeof(struct log_packet)) <
         (ssize_t)sizeof(struct log_packet)) {
            break;
        }

        struct timespec delay;
        tv_delta_to_nano(&stamp, &packet.tv, &delay);
        nsleep_chosen(&delay, &xdelay);
    }

    fwrite("\n", 1, 1, stdout);
    return;
}

//-----------------------------------------------------------------------------
static __attribute__((unused)) unsigned long calc_ovcorr(unsigned long ad,
 int rd) {
    unsigned long av = 0;
    struct timeval a, b;
    struct timespec s;
    int count = rd;

    fprintf(stderr, "Calculating average overhead...");

    while(count--) {
        gettimeofday(&a, NULL);
        s.tv_sec = 0;
        s.tv_nsec = ad;
        nanosleep(&s, NULL);
        gettimeofday(&b, NULL);
        av += 1E6 * (b.tv_sec - a.tv_sec) + b.tv_usec - a.tv_usec;
    }

    av /= rd;
    fprintf(stderr, " %lu s\n", av);
    return av;
}

static int get_options(int *argc, const char ***argv) {
    static const char *_empty_argv[] = {NULL, NULL};
    struct poptOption options_table[] = {
        {NULL, 'A', POPT_ARG_NONE, NULL, 'A', "Use nanosleep() overhead correction", NULL},
        {NULL, 'D', POPT_ARG_NONE, &opt.debug, '\0', "Debug input", NULL},
        {NULL, 'S', POPT_ARG_DOUBLE, &opt.factor, '\0', "Speed factor (default: 1.0)", NULL},
        POPT_AUTOHELP
        POPT_TABLEEND
    };

    poptContext ctx;
    const char **args;
    int c, argk;

    ctx = poptGetContext(**argv, *argc, *argv, options_table, 0);
    while((c = poptGetNextOpt(ctx)) >= 0) {
        switch(c) {
            case 'A':
                //opt.ovcorr = calc_ovcorr(0, 100);
                fprintf(stderr, "Using scheduling overhead correction algorithm\n");
                nsleep_chosen = nsleep_ovcorr;
                break;
        }
    }

    if(c < -1) {
        fprintf(stderr, "%s: %s\n", poptBadOption(ctx, 0), poptStrerror(c));
        poptFreeContext(ctx);
        return 0;
    }

    if((args = poptGetArgs(ctx)) != NULL) {
        argk = SH_count_args(args);
        poptDupArgv(argk, args, argc, argv);
    } else {
        *argc = 0;
        *argv = _empty_argv;
    }

    poptFreeContext(ctx);
    return 1;
}

inline static int imin(int a, int b) {
    return a < b ? a : b;
}

static int nsleep_ovcorr(struct timespec *req, unsigned long *xdelay) {
    /* Sleeps the specified amount of time using an overhead-corrected value.
    This is due to nanosleep() sometimes take one tick longer than it should.
    (See nanosleep(2).) */
    struct timeval sa, sb;
    long dur, over;
    int rv;

    if(*xdelay > 0) {
        long max = req->tv_sec * 1000 + req->tv_nsec / 1000;
        long take = imin(max, *xdelay);
        long long new = req->tv_nsec - ((long long)take * 1000);
        *xdelay -= take;
        if(new < 0) {
            --req->tv_sec;
            req->tv_sec -= -new / NANOSECOND;
            req->tv_nsec = NANOSECOND - new;
        } else {
            req->tv_nsec = new;
        }
    }

    gettimeofday(&sa, NULL);
    rv = nanosleep(req, NULL);
    gettimeofday(&sb, NULL);

    /* Calculate the actual duration and the overhead
    (actual time minus wanted time) */
    dur = MICROSECOND * (sb.tv_sec - sa.tv_sec) + (sb.tv_usec - sa.tv_usec);
    over = dur - (req->tv_sec * 1000 + req->tv_nsec / 1000);
    if(over > 0) { *xdelay += over; }

    return rv;
}

static int nsleep_std(struct timespec *req, unsigned long *xdelay) {
    return nanosleep(req, NULL);
}

inline static void tv_delta_to_nano(const struct timeval *past,
 const struct timeval *now, struct timespec *dest) {
    /* Calculates the time difference between "past" (s) and "now" (s) and
    stores the result in "dest" (ns). */
    unsigned long sec = now->tv_sec - past->tv_sec;
    long acc = now->tv_usec - past->tv_usec;
    if(acc < 0) {
        // past: 1.5, now: 2.0, sec = 2 - 1 = 1, acc = 0 - 500000 = -500000;
        dest->tv_sec = --sec;
        dest->tv_nsec = NANOSECOND + 1000 * acc;
    } else {
        dest->tv_sec = sec;
        dest->tv_nsec = 1000 * acc;
    }
    return;
}

//==[ End of file ]============================================================
