/*=============================================================================
ttyrpld - TTY replay daemon
user/rpld.c - User space daemon (filtering, etc.)
  Copyright (C) Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)
  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/mman.h> // mlock()
#include <sys/stat.h>
#include <sys/time.h> // gettimeofday()
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <popt.h>
#include <pwd.h>
#include <signal.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
extern int posix_memalign(void **, size_t, size_t);
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <libHX.h>
#include "dev.h"
#include "rpl_ioctl.h"
#include "rpl_packet.h"
#include "ushared.h"
#define MAXFNLEN 1024

struct tty {
    uint32_t dev;
    size_t in, out;
    int fd;
};

static void mainloop(int);

static int exclude_packet(uint32_t);
static struct tty *get_tty(uint32_t);
inline static int imin(int, int);
static int log_open(struct tty *);
static void log_write(struct log_packet *, struct tty *, int);
static void log_close(struct tty *);
static int value_cmp(void *, void *);

static int get_options(int *, const char ***);
static int read_config(const char *);
static int pconfig_user(const char *, unsigned char, void *, void *);

static int init_device(const char *);
static int init_sbuffer(int);
static void deinit_sbuffer(void);
static int init_sighandler(void);
static void sighandler_int(int);

// Variables
static struct {
    long bsize, fbsize;
    char *dev, *logdir;
    int sbtty;
    long user_uid;
} opt = {
    bsize: -1,
    fbsize: 16384,
    dev: "/dev/rpl:/dev/misc/rpl",
    logdir: "/var/log/rpl",
    sbtty: 0,
    user_uid: -1,
};

static void *GV_buffer = NULL;
static struct HXbtree *GV_ttys = NULL;
static int GV_running = 1;

//-----------------------------------------------------------------------------
int main(int argc, char **argv) {
    int fd;

    if(!read_config("rpld.conf") ||
     !get_options(&argc, (const char ***)&argv)) {
        return EXIT_FAILURE;
    }

    if((GV_ttys = HXbtree_init(HXBT_ASSOC | HXBT_LKEY | HXBT_PCMP, value_cmp)) == NULL) {
        fprintf(stderr, "HXbtree_init(): %s\n", strerror(errno));
        return EXIT_FAILURE;
    }

    if((fd = init_device(opt.dev)) < 0) {
        fprintf(stderr, "No device could be opened, aborting.\n");
        return EXIT_FAILURE;
    }

    if(init_sbuffer(fd) < 0) { return EXIT_FAILURE; }
    init_sighandler();
    if(opt.user_uid >= 0) { setuid(opt.user_uid); }

    mainloop(fd);
    close(fd);
    deinit_sbuffer();
    return EXIT_SUCCESS;
}

static void mainloop(int fd) {
    while(GV_running) {
        struct log_packet packet;
        struct tty *tty;

        read(fd, &packet.p, sizeof(struct rpld_packet));

        if(packet.p.magic == MAGIC_2_6) {
            // nothing needed
        } else if(packet.p.magic == MAGIC_2_4) {
            packet.p.dev = K26_MKDEV(K24_MAJOR(packet.p.dev),
             K24_MINOR(packet.p.dev));
            packet.p.magic = MAGIC_2_6;
        } else {
            //fprintf(stderr, "Bogus packet!\n");
            continue;
        }

        gettimeofday(&packet.tv, NULL);
        tty = get_tty(packet.p.dev);
        if(packet.p.event == EV_READ) { tty->in += packet.p.size; }
        else if(packet.p.event == EV_WRITE) { tty->out += packet.p.size; }

        if(packet.p.size == 0 || packet.p.event == EV_IOCTL ||
         exclude_packet(packet.p.dev)) {
            lseek(fd, packet.p.size, SEEK_CUR);
            continue;
        }

//        printf("packet: %lu.%lu, ev=%d mgc=%d sz=%hu\n", KD26_PARTS(packet.p.dev), packet.p.event, packet.p.magic, packet.p.size);

        switch(packet.p.event) {
            /* EV_OPEN is not handled since a logfile will automatically be
            opened in log_write(). */
            case EV_READ:
//            case EV_IOCTL:
            case EV_WRITE:
                log_write(&packet, tty, fd);
                break;
            case EV_CLOSE:
                log_close(tty);
                break;
        }
    }

    return;
}

//-----------------------------------------------------------------------------
static int exclude_packet(uint32_t dev) {
    register long maj = K26_MAJOR(dev);
    return maj == 2 /* BSD ptm */ || maj == 128 /* Unix98 ptm */;
}

static struct tty *get_tty(uint32_t dev) {
    struct tty *ret, *tty;
    if((ret = HXbtree_get(GV_ttys, (void *)dev)) != NULL) { return ret; }
    if((tty = malloc(sizeof(struct tty))) == NULL) { return NULL; }
    tty->dev = dev;
    tty->in  = \
    tty->out = 0;
    tty->fd  = -1;
    return HXbtree_add(GV_ttys, (void *)dev, tty)->data;
}

inline static int imin(int a, int b) {
    return a < b ? a : b;
}

static int log_open(struct tty *tty) {
    char buf[MAXFNLEN], devname[64], fmtime[16];
    time_t now = time(NULL);
    struct tm now_tm;

    SH_device_name(tty->dev, devname, 64);
    strftime(fmtime,  16, "%Y%M%d.%H%M%S", localtime_r(&now, &now_tm));

    if(opt.sbtty) {
        snprintf(buf, MAXFNLEN, "%s/%s.%s", opt.logdir, devname, fmtime);
    } else {
        snprintf(buf, MAXFNLEN, "%s/%s.%s", opt.logdir, fmtime, devname);
    }

    return tty->fd = open(buf, O_WRONLY | O_CREAT | O_APPEND, 0600);
}

static void log_write(struct log_packet *packet, struct tty *tty, int fd) {
    int read_left;

    if(tty->fd == -1) { log_open(tty); }
    write(tty->fd, packet, sizeof(struct log_packet));
    read_left = packet->p.size;

    while(read_left > 0) {
        int have_read = read(fd, GV_buffer, imin(read_left, opt.bsize));
        write(tty->fd, GV_buffer, have_read);
        read_left -= have_read;
    }

    return;
}

static void log_close(struct tty *tty) {
    close(tty->fd);
    tty->fd = -1;
    return;
}

static int value_cmp(void *pa, void *pb) {
    return (pa > pb) - (pa < pb);
}

//-----------------------------------------------------------------------------
static int get_options(int *argc, const char ***argv) {
    static const char *_empty_argv[] = {NULL};
    char *tmp;
    struct poptOption options_table[] = {
        {"bsize", 0, POPT_ARG_LONG, &opt.bsize, 0,
         "Buffer size (overrides ioctl()-retrieved size)", "bytes"},
        {"fbsize", 0, POPT_ARG_LONG, &opt.fbsize, 0,
         "Buffer size in case ioctl() fails", "bytes"},
        {"sbtty", 0, POPT_ARG_NONE, &opt.sbtty, 0,
         "Output file names will appear sorted by tty", NULL},
        {NULL, 'D', POPT_ARG_STRING, &opt.dev, 0,
         "Path to the RPL device", "file"},
        {NULL, 'L', POPT_ARG_STRING, &opt.logdir, 0,
         "Directory to put logfiles into", "dir"},
        {NULL, 'U', POPT_ARG_STRING, &tmp, 'U',
         "User to change to", "user"},
        {NULL, 'c', POPT_ARG_STRING, &tmp, 'c',
         "Read configuration from file (on top of hardcoded conf)", "file"},
        POPT_AUTOHELP
        POPT_TABLEEND
    };

    poptContext ctx;
    const char **args;
    int c, argk;

    ctx = poptGetContext(**argv, *argc, *argv, options_table, 0);
    while((c = poptGetNextOpt(ctx)) >= 0) {
        switch(c) {
            case 'U':
                if(!pconfig_user(NULL, 0, tmp, NULL)) { return 0; }
                free(tmp);
                break;
            case 'c':
                read_config(tmp);
                free(tmp);
                break;
        }
    }

    if(c < -1) {
        fprintf(stderr, "%s: %s\n", poptBadOption(ctx, 0), poptStrerror(c));
        poptFreeContext(ctx);
        return 0;
    }

    if((args = poptGetArgs(ctx)) != NULL) {
        argk = SH_count_args(args);
        poptDupArgv(argk, args, argc, argv);
    } else {
        *argc = 0;
        *argv = _empty_argv;
    }

    poptFreeContext(ctx);
    return 1;
}

static int read_config(const char *file) {
    static struct rconfig_opt options_table[] = {
        {"BSIZE",  RCONF_LONG,   &opt.bsize,  NULL},
        {"FBSIZE", RCONF_LONG,   &opt.fbsize, NULL},
        {"DEVICE", RCONF_STRING, &opt.dev,    NULL},
        {"LOGDIR", RCONF_STRING, &opt.logdir, NULL},
        {"SBTTY",  RCONF_IBOOL,  &opt.sbtty,  NULL},
        {"USER",   RCONF_CB,     NULL,        pconfig_user, exit},
    };
    return HX_rconfig(file, options_table);
}

static int pconfig_user(const char *key, unsigned char type, void *ptr,
 void *uptr) {
    struct passwd *ent;
    if((ent = getpwnam(ptr)) == NULL) {
        fprintf(stderr, "No such user: %s\n", (char *)ptr);
        if(uptr != NULL) { exit(EXIT_FAILURE); }
    }
    opt.user_uid = ent->pw_uid;
    return 1;
}

//-----------------------------------------------------------------------------
static int init_device(const char *in_devs) {
    char *copy = HX_strdup(in_devs), *devs = copy, *devp;
    int fd;

    while((devp = HX_strsep(&devs, ":")) != NULL) {
        if((fd = open(devp, O_RDONLY)) >= 0) { return fd; }
        if(errno != ENOENT) {
            fprintf(stderr, "Could not open %s even though it exists: %s\n",
             devp, strerror(errno));
        }
        ++devp;
    }

    free(copy);
    return -1;
}

static int init_sbuffer(int fd) {
    int eax;

    if(opt.bsize > 0) {
        // nothing
    } else if((eax = ioctl(fd, RPL_IOC_GETBUFSIZE, 0)) >= 0) {
        // use ioctl() value if BSIZE is not given
        opt.bsize = eax;
    } else {
        // use FBSIZE if ioctl() failed
        fprintf(stderr, "Warning: Could not detect buffer size from RPL"
         " device\n");
        opt.bsize = opt.fbsize;
    }

    if(posix_memalign(&GV_buffer, sysconf(_SC_PAGESIZE), opt.bsize) != 0) {
        eax = errno;
        perror("malloc()/posix_memalign()");
        return -(errno = eax);
    }

    /* Done so that this memory area can not be swapped out, which is essential
    against buffer overruns. */
    mlock(GV_buffer, opt.bsize);
    return 1;
}

static void deinit_sbuffer(void) {
    munlock(GV_buffer, opt.bsize);
    free(GV_buffer);
    return;
}

static int init_sighandler(void) {
    struct sigaction sig;
    sig.sa_handler = sighandler_int;
    sigemptyset(&sig.sa_mask);
    sig.sa_flags = SA_RESTART;
    return sigaction(SIGINT, &sig, NULL) + sigaction(SIGTERM, &sig, NULL);
}

static void sighandler_int(int s) {
    if(GV_running-- == 0) {
        /*fprintf(stderr, "Second time we received SIGINT/SIGTERM,"
         " exiting immediately.\n");*/
        exit(EXIT_FAILURE);
    }
    //fprintf(stderr, "Received SIGINT/SIGTERM, shutting down.\n");
    GV_running = 0;
    return;
}

//==[ End of file ]============================================================
