/*=============================================================================
ttyrpld - TTY replay daemon
user/replay.c - Realtime TTY log analyzer
  Copyright (C) Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/stat.h>
#include <sys/time.h> // usec def, gettimeofday()
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <popt.h>
#include <sched.h> // sched_yield()
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h> // nsec def, nanosleep()
#include <unistd.h>

#include "dev.h"
#include "rpl_packet.h"
#include "ushared.h"

#define MICROSECOND 1000000
#define NANOSECOND  1000000000
#define BUFSIZE     4096

struct {
    double factor;
    int live, stdsleep;
    long ovcorr;
} Opt = {
    .factor   = 1.0,
    .live     = 0,
    .stdsleep = 0,
};

static void replay_file(int);

static unsigned long calc_ovcorr(unsigned long, int) __attribute__((unused));
static int get_options(int *, const char ***);
inline static int imin(int, int);
static int usleep_std(struct timeval *, long *);
static int usleep_ovcorr(struct timeval *, long *);
static void tv_delta(const struct timeval *, const struct timeval *,
 struct timeval *);
inline static unsigned long long tv2usec(const struct timeval *);
inline static void usec2tv(unsigned long long, struct timeval *);

static int (*usleep_chosen)(struct timeval *, long *) = usleep_ovcorr;

//-----------------------------------------------------------------------------
int main(int argc, char **argv) {
    if(!get_options(&argc, (const char ***)&argv)) { return EXIT_FAILURE; }

    if(Opt.stdsleep) {
        /* Don't do it, it just does not work very nice with high -S speeds
        and short delays. */
        usleep_chosen = usleep_std;
    }

    printf(
      "> ttyreplay\n"
      "This program comes with ABSOLUTELY NO WARRANTY; it is free software\n"
      "and you are welcome to redistribute it under certain conditions;\n"
      "for details see the doc/GPL2.txt file which should have come with\n"
      "this program.\n\n"
    );

    while(*argv != NULL) {
        int fd;
        if((fd = open(*argv, O_RDONLY)) < 0) {
            fprintf(stderr, "Could not open %s: %s\n",
             *argv, strerror(errno));
            ++argv;
            continue;
        }

        replay_file(fd);
        close(fd);
        ++argv;
    }

    return EXIT_SUCCESS;
}

static void replay_file(int fd) {
    struct log_packet packet;
    struct timeval stamp;
    char buf[BUFSIZE];
    long skew = 0;
    ssize_t eax = read(fd, &packet, sizeof(struct log_packet));

    if(eax == (ssize_t)sizeof(struct log_packet)) while(1) {
        size_t have_read, read_left;
        struct timeval delta;

        memcpy(&stamp, &packet.tv, sizeof(struct timeval));
        read_left = packet.p.size;

        while(read_left > 0) {
            have_read = read(fd, buf, imin(read_left, BUFSIZE));
            if(packet.p.event == EV_WRITE) {
                fwrite(buf, have_read, 1, stdout);
            }
            read_left -= have_read;
        }

        fflush(stdout);
        if(read(fd, &packet, sizeof(struct log_packet)) <
         (ssize_t)sizeof(struct log_packet)) {
            break;
        }

        tv_delta(&stamp, &packet.tv, &delta);
        if(Opt.factor != 1.0) {
            usec2tv(tv2usec(&delta) / Opt.factor, &delta);
        }
        usleep_chosen(&delta, &skew);
    }

    fwrite("\n", 1, 1, stdout);
    return;
}

//-----------------------------------------------------------------------------
static unsigned long calc_ovcorr(unsigned long ad, int rd) {
    struct timespec s = {.tv_sec = 0, .tv_nsec = ad};
    struct timeval start, stop;
    unsigned long av = 0;
    int count = rd;

    fprintf(stderr, "Calculating average overhead...");

    while(count--) {
        gettimeofday(&start, NULL);
        nanosleep(&s, NULL);
        gettimeofday(&stop, NULL);
        av += MICROSECOND * (stop.tv_sec - start.tv_sec) +
         stop.tv_usec - start.tv_usec;
    }

    av /= rd;
    fprintf(stderr, " %lu s\n", av);
    return av;
}

static int get_options(int *argc, const char ***argv) {
    static const char *_empty_argv[] = {NULL, NULL};
    struct poptOption options_table[] = {
        {"std", 0, POPT_ARG_NONE, &Opt.stdsleep, 0,
         "Use no nanosleep() overhead correction", NULL},
/*
        {NULL, 'L', POPT_ARG_NONE, NULL, 'L',
         "Live replay (like a `tail -f`)", NULL},
*/
        {NULL, 'S', POPT_ARG_DOUBLE, &Opt.factor, '\0',
         "Speed factor (default: 1.0)", NULL},
/*
        {NULL, 'l', POPT_ARG_NONE, NULL, 'l',
         "Play file as usual and switch into live mode on EOF", NULL},
*/
        POPT_AUTOHELP
        POPT_TABLEEND
    };

    poptContext ctx;
    const char **args;
    int c, argk;

    ctx = poptGetContext(**argv, *argc, *argv, options_table, 0);
    while((c = poptGetNextOpt(ctx)) >= 0) {
        switch(c) {
            case 'A':
                Opt.ovcorr = calc_ovcorr(0, 100);
                fprintf(stderr, "Using scheduling overhead correction algorithm\n");
                usleep_chosen = usleep_ovcorr;
                break;
            case 'L':
                Opt.live = 2;
                break;
            case 'l':
                Opt.live = 1;
                break;
        }
    }

    if(c < -1) {
        fprintf(stderr, "%s: %s\n", poptBadOption(ctx, 0), poptStrerror(c));
        poptFreeContext(ctx);
        return 0;
    }

    if((args = poptGetArgs(ctx)) != NULL) {
        argk = SH_count_args(args);
        poptDupArgv(argk, args, argc, argv);
    } else {
        *argc = 0;
        *argv = _empty_argv;
    }

    poptFreeContext(ctx);
    return 1;
}

inline static int imin(int a, int b) {
    return a < b ? a : b;
}

static int usleep_std(struct timeval *req, long *skew) {
    struct timespec nano_req = {
        .tv_sec  = req->tv_sec,
        .tv_nsec = req->tv_usec * 1000,
    };
    return nanosleep(&nano_req, NULL);
}

static int usleep_ovcorr(struct timeval *req, long *skew) {
    /* The manual page for nanosleep(2) says this under the "BUGS" section:

      >The current implementation of nanosleep is  based  on  the
      >normal  kernel  timer mechanism, which has a resolution of
      >1/HZ s (i.e, 10 ms on Linux/i386 and 1 ms on Linux/Alpha).

    (*) for i386/2.4: HZ=100, i.e. 10 ms (10000 s)
        for  all/2.6: HZ=1000, i.e. 1 ms (1000 s)

      >Therefore, nanosleep pauses always for at least the speci-
      >fied time, however it can take up to  10  ms  longer  than
      >specified  until  the  process becomes runnable again. [...]

      >As some applications  require  much  more  precise  pauses
      >(e.g.,  in  order to control some time-critical hardware),
      >nanosleep is also capable of short high-precision  pauses.
      >If  the process is scheduled under a real-time policy like
      >SCHED_FIFO or SCHED_RR, then pauses of up to 2 ms will  be
      >performed as busy waits with microsecond precision.

    Unfortunately, repeated busy waits lock out all other applications, so
    selecting another scheduling policy is inacceptable for ttyreplay, also
    because selecting another policy is only available to the superuser.

    Thus, I have designed a function that tries to cope with the longer delays
    by waiting a little shorter the next time when the previous call took
    longer. The minimal margin of course still is 1/100 (1/1000) s, why the
    algorithm splits up into multiple cases to make the best performance. They
    are explained below. */

    struct timeval start, stop;
    struct timespec nano_req;
    long long req_usec;
    long dur, over;
    int rv;

    /* The following if() block involves a "lot" of 64-bit calculations,
    which take more time on 32-bit archs. Even if my current AMD runs at
    approx. 30 billion instructions/second, it is only fair to add these few
    nanoseconds to the skew. To be non-plus-ultra fair, they are only added if
    we do not do an early return (case 2 & 3). */
    gettimeofday(&start, NULL);
    req_usec = tv2usec(req);

    if(req_usec > Opt.ovcorr) {
        /* If the user requests a delay which is greater than the minimal delay
        (calculated by calc_ovcorr()), we can simply take away some time from
        the request. */

        if(*skew + Opt.ovcorr <= req_usec) {
            /* If the accumulated skew time plus the minimal delay fits into
            the request, the request is reduced and the skew is zeroed. */
            usec2tv(req_usec -= *skew + Opt.ovcorr, req);
            *skew = 0;
        } else {
            /* There is more skew than the current request, so we can take away
            at most the requested time. (In simple language: We already paused
            enough in the past so we can entirely skip this pause.) */
            *skew -= req_usec;
            return 0;
        }
    } else if(*skew >= -Opt.ovcorr) {
        /* The code is the same as the above case, the intent too, but with a
        specialty: The skew can become negative (i.e. we paused too few) up to
        a specific degree (1/HZ s). */
        *skew -= req_usec;
        return 0;
    }

    /* If none of these three cases, or case 2 applies, nanosleep() will be
    called with the time request. */
    nano_req.tv_sec  = req->tv_sec;
    nano_req.tv_nsec = req->tv_usec * 1000,
    rv = nanosleep(&nano_req, NULL);
    gettimeofday(&stop, NULL);

    /* Calculate the actual duration of nanosleep(), and from that, the
    overhead (actual time minus wanted time) which is added to the skew. */
    dur = MICROSECOND * (stop.tv_sec - start.tv_sec) +
     (stop.tv_usec - start.tv_usec);
    // using req_usec is a little trick to save 64-bit calculations
    over = dur - req_usec;
    *skew += over;

    return rv;
}

static void tv_delta(const struct timeval *past, const struct timeval *now,
 struct timeval *dest) {
    /* Calculates the time difference between "past" and "now" and stores the
    result in "dest". All parameters in s. */
    unsigned long sec = now->tv_sec - past->tv_sec;
    long acc = now->tv_usec - past->tv_usec;

    if(acc < 0) {
        // past: 1.5, now: 2.0, sec = 2 - 1 = 1, acc = 0 - 500000 = -500000;
        dest->tv_sec = --sec;
        dest->tv_usec = MICROSECOND + acc;
    } else {
        dest->tv_sec = sec;
        dest->tv_usec = acc;
    }

    return;
}

inline static unsigned long long tv2usec(const struct timeval *req) {
    return req->tv_sec * MICROSECOND + req->tv_usec;
}

inline static void usec2tv(unsigned long long usec, struct timeval *req) {
    req->tv_sec  = usec / MICROSECOND;
    req->tv_usec = usec % MICROSECOND;
    return;
}

//==[ End of file ]============================================================
