/*=============================================================================
ttyrpld - TTY replay daemon
user/rpld.c - User space daemon (filtering, etc.)
  Copyright (C) Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/mman.h> // mlock()
#include <sys/stat.h>
#include <sys/time.h> // gettimeofday()
#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <popt.h>
#include <pwd.h>
#include <signal.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
extern int posix_memalign(void **, size_t, size_t);
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <libHX.h>
#include "dev.h"
#include "rpl_ioctl.h"
#include "rpl_packet.h"
#include "ushared.h"
#define MAXFNLEN 1024

struct tty {
    uint32_t dev;
    size_t in, out;
    int fd;
};

// The prototype order is the same as the function body order
static void mainloop(int);

static int packet_preaction(struct log_packet *, struct tty *);
static int packet_exclude(struct log_packet *);
static int packet_action(struct log_packet *, struct tty *, int);

static struct tty *get_tty(uint32_t);
static int log_open(struct tty *);
static void log_write(struct log_packet *, struct tty *, int);
static void log_close(struct tty *);

static int init_device(const char *);
static int init_sbuffer(int);
static void deinit_sbuffer(void);
static int init_sighandler(void);
static void sighandler_int(int);
static void sighandler_alrm(int);

static int get_options(int *, const char ***);
static int read_config(const char *);
static int pconfig_user(const char *, unsigned char, void *, void *);

static char *getnamefromuid(uid_t, char *, size_t);
static uid_t getuidfromname(const char *);
inline static int imin(int, int);
static char *replace(char *, const char *, size_t, char **);
static int value_cmp(void *, void *);

// Variables
static struct {
    long bsize, fbsize;
    char *dev, *ofmt;
    int verbose;
    long user_id;
} Opt = {
    .bsize    = -1,
    .fbsize   = 16384,
    .dev      = "/dev/rpl:/dev/misc/rpl",
    .ofmt     = "%u.%d.%l",
    .verbose  = 0,
    .user_id = -1,
};

struct {
    size_t open, read, write, ioctl, close, badpack;
    unsigned long long in, out;
} Stats;

static void *Buffer = NULL;
static struct HXbtree *Ttys = NULL;
static int Running = 1;

//-----------------------------------------------------------------------------
int main(int argc, char **argv) {
    int fd;

    if(!read_config("rpld.conf") ||
     !get_options(&argc, (const char ***)&argv)) {
        return EXIT_FAILURE;
    }

    if(Opt.verbose) {
        memset(&Stats, 0, sizeof(Stats));
        printf(
          "# rpld\n"
          "This program comes with ABSOLUTELY NO WARRANTY; it is free"
          " software\n"
          "and you are welcome to redistribute it under certain conditions;\n"
          "for details see the doc/GPL2.txt file which should have come with\n"
          "this program.\n\n"
        );
    }

    if((Ttys = HXbtree_init(HXBT_ASSOC | HXBT_LKEY | HXBT_FCMP,
     value_cmp)) == NULL) {
        perror("HXbtree_init()");
        return EXIT_FAILURE;
    }

    if((fd = init_device(Opt.dev)) < 0) {
        fprintf(stderr, "No device could be opened, aborting.\n");
        return EXIT_FAILURE;
    }

    if(init_sbuffer(fd) < 0) { return EXIT_FAILURE; }
    init_sighandler();
    if(Opt.verbose) { alarm(1); } // Callback function for statistics

    if(Opt.user_id >= 0) {
        // Not seteuid(), because it shall not be posibble to regain privs.
        setuid(Opt.user_id);
    }

    mainloop(fd);
    close(fd);
    deinit_sbuffer();
    return EXIT_SUCCESS;
}

static void mainloop(int fd) {
    while(Running) {
        struct log_packet packet;
        struct tty *tty;

        read(fd, &packet.p, sizeof(struct rpld_packet));

        if(packet.p.magic == MAGIC_2_6) {
            // nothing needed
        } else if(packet.p.magic == MAGIC_2_4) {
            packet.p.dev = K26_MKDEV(K24_MAJOR(packet.p.dev),
             K24_MINOR(packet.p.dev));
            packet.p.magic = MAGIC_2_6;
        } else {
            ++Stats.badpack;
            if(Opt.verbose) {
                printf("\n" "Bogus packet (magic is 0x%X)!\n",
                 packet.p.magic);
            }
            continue;
        }

        /* Timestamp is taken here instead of in the Kernel module to reduce
        the amount of data which has to be copied over the device. */
        gettimeofday(&packet.tv, NULL);
        if((tty = get_tty(packet.p.dev)) == NULL ||
         !packet_preaction(&packet, tty) || packet_exclude(&packet)) {
            lseek(fd, packet.p.size, SEEK_CUR);
            continue;
        }

        packet_action(&packet, tty, fd);
    }

    return;
}

//-----------------------------------------------------------------------------
static int packet_preaction(struct log_packet *packet, struct tty *tty) {
    switch(packet->p.event) {
        case EV_OPEN:
            ++Stats.open;
            break;
        case EV_READ:
            ++Stats.read;
            Stats.in += packet->p.size;
            tty->in  += packet->p.size;
            break;
        case EV_WRITE:
            ++Stats.write;
            Stats.out += packet->p.size;
            tty->out  += packet->p.size;
            break;
        case EV_IOCTL:
            ++Stats.ioctl;
            break;
        case EV_CLOSE:
            ++Stats.close;
            break;
        default:
            if(Opt.verbose) {
                printf("\n" "Unknown packet type 0x%X\n",
                 packet->p.event);
            }
            return 0;
    }
    return 1;
}

static int packet_exclude(struct log_packet *packet) {
    register long maj;
    if(packet->p.event == EV_IOCTL) {
        /* ioctl requests are actually logged by the rpldev Kernel module,
        but I do not see a use for them ATM. */
        return 1;
    }

    /* pty masters are an exact copy of their slaves,
    so no need to record them too. */
    maj = K26_MAJOR(packet->p.dev);
    return maj == 2   /* BSD ptm (/dev/ptypXY) */ ||
           maj == 128 /* Unix98 ptm (-)        */;
}

static int packet_action(struct log_packet *packet, struct tty *tty, int fd) {
    switch(packet->p.event) {
       /* EV_OPEN is not handled since a logfile will automatically be opened
       in log_write() if necessary. */
       case EV_READ:
       case EV_WRITE:
           log_write(packet, tty, fd);
           break;
       case EV_CLOSE:
           log_close(tty);
           break;
    }
    return 1;
}

//-----------------------------------------------------------------------------
static struct tty *get_tty(uint32_t dev) {
    struct HXbtree_node *ts;
    struct tty *ret, *tty;
    if((ret = HXbtree_get(Ttys, (void *)dev)) != NULL) { return ret; }
    if((tty = malloc(sizeof(struct tty))) == NULL) { return NULL; }
    tty->dev = dev;
    tty->in  = \
    tty->out = 0;
    tty->fd  = -1;
    if((ts = HXbtree_add(Ttys, (void *)dev, tty)) == NULL) {
        perror("Memory allocation failure");
        return NULL;
    }
    return ts->data;
}

static int log_open(struct tty *tty) {
    char buf[MAXFNLEN], devname[64], fmday[16], fmtime[16], user[32];
    char *subst_catalog[] = {
        "d", fmday,
        "l", devname,
        "t", fmtime,
        "u", user,
        NULL
    };

    { // User
      struct stat sb;
      SH_devname_fs(tty->dev, devname, 64);
      if(stat(devname, &sb) != 0) {
          strcpy(user, "NONE");
      } else if(getnamefromuid(sb.st_uid, user, 32) == NULL) {
          snprintf(user, 32, "%d", sb.st_uid);
      }
    }

    { // Time
      time_t now = time(NULL);
      struct tm now_tm;
      localtime_r(&now, &now_tm);
      strftime(fmday,  16, "%Y%M%d", &now_tm);
      strftime(fmtime, 16, "%H%M%S", &now_tm);
    }

    // Simple tty name
    SH_devname(tty->dev, devname, 64);
    replace(buf, Opt.ofmt, MAXFNLEN, subst_catalog);

    if((tty->fd = open(buf, O_WRONLY | O_CREAT | O_APPEND, 0600)) < 0) {
        fprintf(stderr, "Could not open %s: %s\n", buf, strerror(errno));
    }
    return tty->fd;
}

static void log_write(struct log_packet *packet, struct tty *tty, int fd) {
    int read_left;

    if(tty->fd == -1) { log_open(tty); }
    write(tty->fd, packet, sizeof(struct log_packet));
    read_left = packet->p.size;

    while(read_left > 0) {
        int have_read = read(fd, Buffer, imin(read_left, Opt.bsize));
        write(tty->fd, Buffer, have_read);
        read_left -= have_read;
    }

    return;
}

static void log_close(struct tty *tty) {
    close(tty->fd);
    tty->fd = -1;
    return;
}

//-----------------------------------------------------------------------------
static int init_device(const char *in_devs) {
    char *copy = HX_strdup(in_devs), *devs = copy, *devp;
    int fd;

    while((devp = HX_strsep(&devs, ":")) != NULL) {
        if((fd = open(devp, O_RDONLY)) >= 0) {
            if(Opt.verbose) { printf("Connected to %s\n", devp); }
            free(copy);
            return fd;
        }
        if(errno != ENOENT) {
            fprintf(stderr, "Could not open %s even though it exists: %s\n",
             devp, strerror(errno));
        }
        ++devp;
    }

    free(copy);
    return -1;
}

static int init_sbuffer(int fd) {
    int eax;

    if(Opt.bsize > 0) {
        // nothing
    } else if((eax = ioctl(fd, RPL_IOC_GETBUFSIZE, 0)) >= 0) {
        // use ioctl() value if BSIZE is not given
        Opt.bsize = eax;
    } else {
        // use FBSIZE if ioctl() failed
        fprintf(stderr, "Warning: Could not detect buffer size from RPL"
         " device\n");
        Opt.bsize = Opt.fbsize;
    }

    if(posix_memalign(&Buffer, sysconf(_SC_PAGESIZE), Opt.bsize) != 0) {
        eax = errno;
        perror("malloc()/posix_memalign()");
        return -(errno = eax);
    }

    /* Done so that this memory area can not be swapped out, which is essential
    against buffer overruns. */
    mlock(Buffer, Opt.bsize);
    return 1;
}

static void deinit_sbuffer(void) {
    munlock(Buffer, Opt.bsize);
    free(Buffer);
    return;
}

static int init_sighandler(void) {
    struct sigaction s_int, s_alrm;
    s_int.sa_handler = sighandler_int;
    s_int.sa_flags   = SA_RESTART;
    sigemptyset(&s_int.sa_mask);

    s_alrm.sa_handler = sighandler_alrm;
    s_alrm.sa_flags   = SA_RESTART;
    sigemptyset(&s_alrm.sa_mask);
    
    return sigaction(SIGINT, &s_int, NULL) + sigaction(SIGTERM, &s_int, NULL) +
     sigaction(SIGALRM, &s_alrm, NULL);
}

static void sighandler_int(int s) {
    if(Running-- == 0) {
        if(Opt.verbose) {
            printf("Second time we received SIGINT/SIGTERM,"
             " exiting immediately.\n");
        }
        exit(EXIT_FAILURE);
    }
    if(Opt.verbose) {
        printf("\nReceived SIGINT/SIGTERM, shutting down.\n");
    }
    Running = 0;
    return;
}

static void sighandler_alrm(int s) {
    fprintf(stderr, "\r" "opn/cls: %zu/%zu  r/w: %zu/%zu (%llu/%llu)"
      " ioctl: %zu  bad: %zu",
      Stats.open, Stats.close, Stats.read, Stats.write, Stats.in,
      Stats.out, Stats.ioctl, Stats.badpack
    );
    if(Opt.verbose) { alarm(1); }
    return;
}

//-----------------------------------------------------------------------------
static int get_options(int *argc, const char ***argv) {
    static const char *_empty_argv[] = {NULL};
    char *tmp;
    struct poptOption options_table[] = {
        {"bsize", 0, POPT_ARG_LONG, &Opt.bsize, 0,
         "Buffer size (overrides ioctl()-retrieved size)", "bytes"},
        {"fbsize", 0, POPT_ARG_LONG, &Opt.fbsize, 0,
         "Buffer size in case ioctl() fails", "bytes"},
        {NULL, 'D', POPT_ARG_STRING, &Opt.dev, 0,
         "Path to the RPL device", "file"},
        {NULL, 'O', POPT_ARG_STRING, &Opt.ofmt, 0,
         "Override OFMT variable", "string"},
        {NULL, 'U', POPT_ARG_STRING, &tmp, 'U',
         "User to change to", "user"},
        {NULL, 'c', POPT_ARG_STRING, &tmp, 'c',
         "Read configuration from file (on top of hardcoded conf)", "file"},
        {NULL, 'v', POPT_ARG_NONE, &Opt.verbose, 0,
         "Print statistics while rpld is running", NULL},
        POPT_AUTOHELP
        POPT_TABLEEND
    };

    poptContext ctx;
    const char **args;
    int c, argk;

    ctx = poptGetContext(**argv, *argc, *argv, options_table, 0);
    while((c = poptGetNextOpt(ctx)) >= 0) {
        switch(c) {
            case 'U':
                if(!pconfig_user(NULL, 0, tmp, NULL)) { return 0; }
                free(tmp);
                break;
            case 'c':
                read_config(tmp);
                free(tmp);
                break;
        }
    }

    if(c < -1) {
        fprintf(stderr, "%s: %s\n", poptBadOption(ctx, 0), poptStrerror(c));
        poptFreeContext(ctx);
        return 0;
    }

    if((args = poptGetArgs(ctx)) != NULL) {
        argk = SH_count_args(args);
        poptDupArgv(argk, args, argc, argv);
    } else {
        *argc = 0;
        *argv = _empty_argv;
    }

    poptFreeContext(ctx);
    return 1;
}

static int read_config(const char *file) {
    static struct rconfig_opt options_table[] = {
        {"BSIZE",  RCONF_LONG,   &Opt.bsize,  NULL},
        {"FBSIZE", RCONF_LONG,   &Opt.fbsize, NULL},
        {"DEVICE", RCONF_STRING, &Opt.dev,    NULL},
        {"OFMT",   RCONF_STRING, &Opt.ofmt,   NULL},
        {"USER",   RCONF_CB,     NULL,        pconfig_user},
        {NULL},
    };
    return HX_rconfig(file, options_table);
}

static int pconfig_user(const char *key, unsigned char type, void *ptr,
 void *uptr) {
    if((Opt.user_id = getuidfromname(ptr)) < 0) {
        fprintf(stderr, "No such user: %s\n", (const char *)ptr);
        exit(EXIT_FAILURE);
    }
    return 1;
}

//-----------------------------------------------------------------------------
static char *getnamefromuid(uid_t uid, char *result, size_t len) {
    struct passwd ent, *ep;
    char additional[1024];
    getpwuid_r(uid, &ent, additional, 1024, &ep);
    if(ep == NULL) { return NULL; }
    strncpy(result, ep->pw_name, len - 1);
    result[len - 1] = '\0';
    return result;
}

static uid_t getuidfromname(const char *name) {
    struct passwd ent, *ep;
    char additional[1024];
    getpwnam_r(name, &ent, additional, 1024, &ep);
    if(ep == NULL) { return -1; }
    return ep->pw_uid;
}

inline static int imin(int a, int b) {
    return a < b ? a : b;
}

static char *replace(char *dest, const char *src, size_t count,
 char **catalog) {
    char *destp = dest;
    *destp = '\0';
    destp[count - 1] = '\0';

    while(*src != '\0' && count > 1) {
        size_t max = imin(strcspn(src, "%"), count - 1);
        char **catp = catalog, *key, *value;
        int got_spec = 0;

        strncpy(destp, src, max);
        src   += max;
        destp += max;
        count -= max;
        if(*src != '%' || count < 2) { break; }
        ++src;
        if(*src == '%') { // "%%"
            *destp++ = '%';
            if(--count < 2) { break; }
            ++src;
            continue;
        }
        if((unsigned char)*src >= 128 || !isalpha(*src)) {
            fprintf(stderr, "%% without specifier in OFMT.\n");
            break;
        }

        while((key = *catp++) != NULL) {
            value = *catp++;
            if(*src == *key) {
                strncpy(destp, value, count - 1);
                max = strlen(destp);
                destp += max;
                count -= max;
                ++got_spec;
                break;
            }
        }
        if(count < 2) { break; }
        if(!got_spec) { fprintf(stderr, "Invalid specifier: %%%c\n", *src); }
        ++src;
    }
    return dest;
}

static int value_cmp(void *pa, void *pb) {
    return (pa > pb) - (pa < pb);
}

//==[ End of file ]============================================================
