/*=============================================================================
ttyrpld - TTY replay daemon
user/shared.c - Shared data
  Copyright (C) Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/types.h>
#include <stdio.h>
#include "dev.h"
#include "ushared.h"

static const char *m_idx = "pqrstuvwxyzabcdef", *s_idx = "0123456789abcdef";

//-----------------------------------------------------------------------------
int SH_count_args(const char **args) {
    int argk = 0;
    while(*args++ != NULL) { ++argk; }
    return argk;
}

char *SH_devname(uint32_t dev, char *buf, size_t count) {
    /* This function returns a "simple name" which can be integrated into
    rpld's log file name scheme. */
    unsigned long minor = K26_MINOR(dev);

    switch(K26_MAJOR(dev)) {
        case 2: // BSD pty masters
            snprintf(buf, count, "pty%c%c",
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 3: // BSD pty slaves
            snprintf(buf, count, "tty%c%c",
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 4:
            // Virtual console (0-63) and serial console (64-255)
            if(minor < 64) { snprintf(buf, count, "vc-%lu", minor); }
            else { snprintf(buf, count, "ttyS%lu", minor - 64); }
            break;
        case 5:
            if(minor == 1) { snprintf(buf, count, "console"); }
            else { goto unknown; }
        case 128:
            // Unix98 pty masters (got no real /dev entry, though, see below)
            snprintf(buf, count, "ptm-%lu", minor);
            break;
        case 136: // Unix98 pty slaves
            snprintf(buf, count, "pts-%lu", minor);
            break;
    }
    return buf;

unknown:
    /* Not every char device is a tty, and I even do not want to include a
    name database for all possible tty devices. */
    snprintf(buf, count, "%lu-%lu", K26_MAJOR(dev), minor);
    return buf;
}

char *SH_devname_fs(uint32_t dev, char *buf, size_t count) {
    /* This function returns a device filename generated from major/minor
    number, which is then used for stat(). */
    unsigned long minor = K26_MINOR(dev);

    switch(K26_MAJOR(dev)) {
        case 2: // BSD pty masters
            snprintf(buf, count, "/dev/pty%c%c",
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 3: // BSD pty slaves
            snprintf(buf, count, "/dev/tty%c%c",
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 4:
            // Virtual console (0-63) and serial console (64-255)
            if(minor < 64) { snprintf(buf, count, "/dev/tty%lu", minor); }
            else { snprintf(buf, count, "/dev/ttyS%lu", minor - 64); }
            break;
        case 5:
            if(minor == 1) { snprintf(buf, count, "/dev/console"); }
            else { goto unknown; }
        case 128:
            /* Unix98 pty masters. They usually have no device nodes in /dev,
            and should instead be accessed over the multiplexer /dev/ptmx.
            However IIRC, devfs popped them up in /dev/ptm. */
            snprintf(buf, count, "/dev/ptm/%lu", minor);
            break;
        case 136: // Unix98 pty slaves
            snprintf(buf, count, "/dev/pts/%lu", minor);
            break;
    }
    return buf;

unknown:
    return NULL;
}

//==[ End of file ]============================================================
