/*=============================================================================
ttyrpld - TTY replay daemon
user/rpld.c - User space daemon (filtering, etc.)
  Copyright (C) Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/mman.h> // mlock()
#include <sys/stat.h>
#include <sys/time.h> // gettimeofday()
#include <sys/types.h>
#include <ctype.h>
#include <endian.h>
#include <errno.h>
#include <fcntl.h>
#include <popt.h>
#include <pwd.h>
#include <signal.h>
#include <stdarg.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
extern int posix_memalign(void **, size_t, size_t);
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include <libHX.h>
#include "dev.h"
#include "rpl_ioctl.h"
#include "rpl_packet.h"
#include "ushared.h"

enum {
    NF_NEWLINE = 1 << 0,
};

struct tty {
    uint32_t dev;
    int fd;
};

// The prototype order is the same as the function body order
static void mainloop(int);

static int packet_preaction(struct mem_packet *);
static int packet_action(struct mem_packet *, struct tty *, int);

static char *devname(uint32_t, char *, size_t);
static char *devname_fs(uint32_t, char *, size_t);
static struct tty *get_tty(uint32_t);
static int log_open(struct tty *);
static void log_write(struct mem_packet *, struct tty *, int);
static void log_close(struct tty *);

static int init_device(const char *);
static int init_sbuffer(int);
static void deinit_sbuffer(void);
static const char *mknod_rpldev(void);
static int init_sighandler(void);
static void sighandler_int(int);
static void sighandler_alrm(int);

static int get_options(int *, const char ***);
static int read_config(const char *);
static int read_config_bp(const char *, const char *);
static int pconfig_user(const char *, unsigned char, void *, void *);

static char *getnamefromuid(uid_t, char *, size_t);
static uid_t getuidfromname(const char *);
static int notify(int, unsigned int, const char *, ...);
static char *replace(char *, const char *, size_t, char **);
static int value_cmp(void *, void *);

// Global variables (begin with uppercase character)
static struct {
    char *_version, *ovdev, *fbdev, *ofmt;
    int nocreat, syslog, verbose;
    long bsize, user_id;
} Opt = {
    ._version = "0.85",
    .bsize    = 16384,
    .ovdev    = NULL,
    .fbdev    = "/dev/rpl:/dev/misc/rpl",
    .nocreat  = 0,
    .ofmt     = "%u@%d.%t.%l",
    .syslog   = 0,
    .verbose  = 0,
    .user_id  = -1,
};

struct {
    size_t open, read, write, ioctl, close, badpack;
    unsigned long long in, out;
} Stats;

static struct HXbtree *Ttys = NULL;
static void *Buffer = NULL;
static int Running = 1;

//-----------------------------------------------------------------------------
int main(int argc, const char **argv) {
    int fd;

    if(!read_config_bp(*argv, "rpld.conf") || !get_options(&argc, &argv)) {
        return EXIT_FAILURE;
    }

    if(Opt.verbose) {
        memset(&Stats, 0, sizeof(Stats));
        printf(
          "# rpld %s\n"
          "This program comes with ABSOLUTELY NO WARRANTY; it is free"
          " software\n"
          "and you are welcome to redistribute it under certain conditions;\n"
          "for details see the doc/GPL2.txt file which should have come with\n"
          "this program.\n\n",
          Opt._version
        );
    }

    SH_check_balign();
    if((Ttys = HXbtree_init(HXBT_ASSOC | HXBT_LKEY | HXBT_FCMP,
     value_cmp)) == NULL) {
        perror("HXbtree_init()");
        return EXIT_FAILURE;
    }

    if((fd = init_device((Opt.ovdev != NULL) ? Opt.ovdev : Opt.fbdev)) < 0) {
        fprintf(stderr, "No device could be opened, aborting.\n");
        return EXIT_FAILURE;
    }

    if(init_sbuffer(fd) < 0) { return EXIT_FAILURE; }
    init_sighandler();

    // Callback tick for statistics
    if(Opt.verbose) { alarm(1); }

    // Not seteuid(), because it shall not be posibble to regain privs.
    if(Opt.user_id >= 0) { setuid(Opt.user_id); }

    if(Opt.syslog) { openlog("rpld", LOG_PID, LOG_DAEMON); }

    mainloop(fd);
    close(fd);
    deinit_sbuffer();
    return EXIT_SUCCESS;
}

static void mainloop(int fd) {
    ssize_t (*reader)(int, void *, size_t) = read;
    int pk = 0;

    { /* read() on the device might return less than we request, so we use
      SH_read_wait() and loop until there is data. However, we do not want to
      wait on e.g. pipes. */
      struct stat sb;
      fstat(fd, &sb);
      if(S_ISCHR(sb.st_mode)) { reader = SH_read_wait; }
    }

    while(Running) {
        struct mem_packet packet;
        struct tty *tty;

        errno = 0;
        if(reader(fd, &packet, sizeof(struct rpld_packet)) <
         sizeof(struct rpld_packet)) {
            Running = 0;
            break;
        }
++pk;
        if(packet.magic == MAGIC_2_6) {
            // nothing needed
        } else if(packet.magic == MAGIC_2_4) {
            packet.magic = MAGIC_2_6;
            packet.dev   = K26_MKDEV(K24_MAJOR(packet.dev),
             K24_MINOR(packet.dev));
        } else {
            ++Stats.badpack;
            notify(LOG_WARNING, NF_NEWLINE, "Bogus packet (magic is 0x%X) pack%d!\n",
             packet.magic, pk);
            continue;
        }

        /* Timestamp is taken here instead of in the Kernel module to reduce
        the amount of data which has to be copied over the device. */
        gettimeofday(&packet.tv, NULL);

        if(!packet_preaction(&packet) || (tty = get_tty(packet.dev)) == NULL) {
            SH_skip(fd, packet.size, 0);
            continue;
        }

        packet_action(&packet, tty, fd);
    }

    if(Opt.verbose) {
        if(errno == 0) { fprintf(stderr, "<EOF on input>\n"); }
        else { perror("read()"); }
    }
    return;
}

//-----------------------------------------------------------------------------
static int packet_preaction(struct mem_packet *packet) {
    /* pty masters are an exact copy of their slaves,
    so no need to record them too. */
    register long maj = K26_MAJOR(packet->dev);

    if(maj == 2 /* BSD ptm (/dev/ptypXY) */) {
        if(packet->event != EV_CLOSE) { return 0; }
        packet->dev = K26_MKDEV(K26_MAJOR(3), K26_MINOR(packet->dev));
    }

    if(maj >= 128 && maj < 136 /* Unix98 ptm (-) */) {
        if(packet->event != EV_CLOSE) { return 0; }
        /* When closing an xterm window or ssh session, the EV_CLOSE event
        actually happens on the master side. To keep it easy, it's device
        signature is simply changed to the slave. */
        packet->dev = K26_MKDEV(K26_MAJOR(packet->dev) + 8,
         K26_MINOR(packet->dev));
    }

    /* The excluded ptm packets above are not counted at all, whereas
    "ignored" packets are. See below for ignore rules. */

    switch(packet->event) {
        case EV_OPEN:
            ++Stats.open;
            break;
        case EV_READ:
            ++Stats.read;
            Stats.in += packet->size;
            break;
        case EV_WRITE:
            ++Stats.write;
            Stats.out += packet->size;
            break;
        case EV_IOCTL:
            ++Stats.ioctl;
            break;
        case EV_CLOSE:
            ++Stats.close;
            break;
        default:
            notify(LOG_WARNING, NF_NEWLINE, "Unknown packet type 0x%X\n",
             packet->event);
            return 0;
    }

    if(packet->event == EV_IOCTL) {
        /* ioctl requests are actually logged by the rpldev Kernel module,
        but I do not see a use for them ATM. */
        return 0;
    }

    return 1;
}

static int packet_action(struct mem_packet *packet, struct tty *tty, int fd) {
    switch(packet->event) {
       /* EV_OPEN is not handled since a logfile will automatically be opened
       in log_write() if necessary. */
       case EV_READ:
       case EV_WRITE:
           log_write(packet, tty, fd);
           break;
       case EV_CLOSE:
           log_close(tty);
           break;
    }
    return 1;
}

//-----------------------------------------------------------------------------
static const char *m_idx = "pqrstuvwxyzabcdef", *s_idx = "0123456789abcdef";

static char *devname(uint32_t dev, char *buf, size_t count) {
    /* This function returns a "simple name" which can be integrated into
    rpld's log file name scheme. */
    unsigned long minor = K26_MINOR(dev);

    switch(K26_MAJOR(dev)) {
        case 2: // BSD pty masters
            snprintf(buf, count, "pty%c%c",
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 3: // BSD pty slaves
            snprintf(buf, count, "tty%c%c",
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 4:
            // Virtual console (0-63) and serial console (64-255)
            if(minor < 64) { snprintf(buf, count, "vc-%lu", minor); }
            else { snprintf(buf, count, "ttyS%lu", minor - 64); }
            break;
        case 5:
            if(minor == 1) { snprintf(buf, count, "console"); }
            else { goto unknown; }
        case 128:
            // Unix98 pty masters (got no real /dev entry, though, see below)
            snprintf(buf, count, "ptm-%lu", minor);
            break;
        case 136: // Unix98 pty slaves
            snprintf(buf, count, "pts-%lu", minor);
            break;
    }
    return buf;

unknown:
    /* Not every char device is a tty, and I even do not want to include a
    name database for all possible tty devices. */
    snprintf(buf, count, "%lu-%lu", K26_MAJOR(dev), minor);
    return buf;
}

static char *devname_fs(uint32_t dev, char *buf, size_t count) {
    /* This function returns a device filename generated from major/minor
    number, which is then used for stat(). */
    unsigned long minor = K26_MINOR(dev);

    switch(K26_MAJOR(dev)) {
        case 2: // BSD pty masters
            snprintf(buf, count, "/dev/pty%c%c",
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 3: // BSD pty slaves
            snprintf(buf, count, "/dev/tty%c%c",
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 4:
            // Virtual console (0-63) and serial console (64-255)
            if(minor < 64) { snprintf(buf, count, "/dev/tty%lu", minor); }
            else { snprintf(buf, count, "/dev/ttyS%lu", minor - 64); }
            break;
        case 5:
            if(minor == 1) { snprintf(buf, count, "/dev/console"); }
            else { goto unknown; }
        case 128:
            /* Unix98 pty masters. They usually have no device nodes in /dev,
            and should instead be accessed over the multiplexer /dev/ptmx.
            However IIRC, devfs popped them up in /dev/ptm. */
            snprintf(buf, count, "/dev/ptm/%lu", minor);
            break;
        case 136: // Unix98 pty slaves
            snprintf(buf, count, "/dev/pts/%lu", minor);
            break;
    }
    return buf;

unknown:
    return NULL;
}

static struct tty *get_tty(uint32_t dev) {
    struct HXbtree_node *ts;
    struct tty *ret, *tty;
    if((ret = HXbtree_get(Ttys, (void *)dev)) != NULL) { return ret; }
    if((tty = malloc(sizeof(struct tty))) == NULL) { return NULL; }
    tty->dev = dev;
    tty->fd  = -1;
    if((ts = HXbtree_add(Ttys, (void *)dev, tty)) == NULL) {
        notify(LOG_ERR, NF_NEWLINE, "Memory allocation failure");
        return NULL;
    }
    return ts->data;
}

static int log_open(struct tty *tty) {
    char buf[MAXFNLEN], devnode[64], fmday[16], fmtime[16], user[32];
    char *subst_catalog[] = {
        "d", fmday,
        "l", devnode,
        "t", fmtime,
        "u", user,
        NULL
    };

    { // User
      struct stat sb;
      devname_fs(tty->dev, devnode, 64);
      if(stat(devnode, &sb) != 0) {
          strcpy(user, "NONE");
      } else if(getnamefromuid(sb.st_uid, user, 32) == NULL) {
          snprintf(user, 32, "%d", sb.st_uid);
      }
    }

    { // Time
      time_t now = time(NULL);
      struct tm now_tm;
      localtime_r(&now, &now_tm);
      strftime(fmday,  16, "%Y%m%d", &now_tm);
      strftime(fmtime, 16, "%H%M%S", &now_tm);
    }

    // Simple tty name
    devname(tty->dev, devnode, 64);
    replace(buf, Opt.ofmt, MAXFNLEN, subst_catalog);

    if(Opt.nocreat) { snprintf(buf, MAXFNLEN, "/dev/null"); }
    if((tty->fd = open(buf, O_WRONLY | O_CREAT | O_APPEND, 0600)) < 0) {
        notify(LOG_ERR, NF_NEWLINE, "Could not open %s: %s\n",
         buf, strerror(errno));
    }

    // ident header (optional)
    if(tty->fd >= 0) {
        struct disk_packet p = {
            .event = EV_IDENT,
            .magic = MAGIC_SIG,
            .tv    = {-1, -1},
        };
        size_t s = snprintf(buf, MAXFNLEN, "ttyrpld %s", Opt._version);
        p.size = s;
#ifdef BIGENDIAN
        SH_swab(&p.size, sizeof(uint16_t));
#endif
        write(tty->fd, &p, sizeof(struct disk_packet));
        write(tty->fd, buf, s);
    }

    return tty->fd;
}

static void log_write(struct mem_packet *packet, struct tty *tty, int fd) {
    size_t s = packet->size;
    int read_left;

    if(tty->fd == -1) { log_open(tty); }
#ifdef BIGENDIAN
    SH_swab(&packet->size, sizeof(packet->size));
    SH_swab(&packet->tv.tv_sec, sizeof(packet->tv.tv_sec));
    SH_swab(&packet->tv.tv_usec, sizeof(packet->tv.tv_usec));
#endif
    write(tty->fd, &packet->size, sizeof(struct disk_packet));
    read_left = s;

    while(read_left > 0) {
        int have_read = read(fd, Buffer, min_uint(read_left, Opt.bsize));
        write(tty->fd, Buffer, have_read);
        read_left -= have_read;
    }

    return;
}

static void log_close(struct tty *tty) {
    close(tty->fd);
    tty->fd = -1;
    HXbtree_del(Ttys, tty);
    return;
}

//-----------------------------------------------------------------------------
static int init_device(const char *in_devs) {
    const char *new_dev = NULL;
    int fd = -1;

    if(Opt.ovdev == NULL && (new_dev = mknod_rpldev()) != NULL) {
        if((fd = open(new_dev, O_RDONLY)) == -1) {
            new_dev = NULL;
            perror("Could not connect to RPL device");
        } else if(Opt.verbose) {
            printf("Connected to RPL device\n");
        }
        unlink(new_dev);
    }

    if(new_dev == NULL) {
        /* No, this will not change into an else if, because new_dev might be
        set to something else above! */
        char *copy = HX_strdup(in_devs), *devs = copy, *devp;

        while((devp = HX_strsep(&devs, ":")) != NULL) {
            if((fd = open(devp, O_RDONLY)) >= 0) {
                if(Opt.verbose) { printf("Connected to %s\n", devp); }
                break;
            }
            if(errno != ENOENT) {
                fprintf(stderr, "Could not open %s even though it exists: %s\n"
                 "Trying next device...\n", devp, strerror(errno));
            }
            ++devp;
        }
        free(copy);
    }

    return fd;
}

static int init_sbuffer(int fd) {
    int eax;

    if((eax = ioctl(fd, RPL_IOC_GETBUFSIZE, 0)) >= 0) {
        Opt.bsize = eax;
    }

    if(posix_memalign(&Buffer, sysconf(_SC_PAGESIZE), Opt.bsize) != 0) {
        eax = errno;
        perror("malloc()/posix_memalign()");
        return -(errno = eax);
    }

    /* Done so that this memory area can not be swapped out, which is essential
    against buffer overruns. */
    mlock(Buffer, Opt.bsize);
    return 1;
}

static void deinit_sbuffer(void) {
    munlock(Buffer, Opt.bsize);
    free(Buffer);
    return;
}

static const char *mknod_rpldev(void) {
    FILE *fp = fopen("/proc/misc", "r");
    int32_t minor = -1;
    char buf[80];

    if(fp == NULL) {
        perror("Could not open /proc/misc");
        return NULL;
    }

    while(fgets(buf, 80, fp) != NULL) {
        char *ptr = buf, *name;
        while(!isdigit(*ptr)) { ++ptr; }
        minor = strtoul(ptr, NULL, 0);
        while(isdigit(*ptr)) { ++ptr; }
        while(isspace(*ptr)) { ++ptr; }

        name = ptr;
        while(isalnum(*ptr)) { ++ptr; }
        *ptr = '\0';
        if(strcmp(name, "rpl") == 0) { break; }
        minor = -1;
    }

    if(minor == -1) {
        printf("RPL module does not seem to be loaded\n");
        return NULL;
    }

    if(mknod(".rpl", S_IFCHR | 0400, GLIBC_MKDEV(10, minor)) != 0) {
        perror("mknod()");
        return NULL;
    }

    return ".rpl";
}

static int init_sighandler(void) {
    struct sigaction s_int, s_alrm;
    s_int.sa_handler = sighandler_int;
    s_int.sa_flags   = SA_RESTART;
    sigemptyset(&s_int.sa_mask);

    s_alrm.sa_handler = sighandler_alrm;
    s_alrm.sa_flags   = SA_RESTART;
    sigemptyset(&s_alrm.sa_mask);

    /* All sigactions() shall be executed, however, if one fails, this function
    shall return <= 0, otherwise >0 upon success.
    Geesh, I love these constructs. */
    return !(!!sigaction(SIGINT, &s_int, NULL) + 
     !!sigaction(SIGTERM, &s_int, NULL) +
     !!sigaction(SIGALRM, &s_alrm, NULL));
}

static void sighandler_int(int s) {
    if(Running-- == 0) {
        if(Opt.verbose) {
            printf("Second time we received SIGINT/SIGTERM,"
             " exiting immediately.\n");
        }
        exit(EXIT_FAILURE);
    }
    if(Opt.verbose) {
        printf("\nReceived SIGINT/SIGTERM, shutting down.\n");
    }
    Running = 0;
    return;
}

static void sighandler_alrm(int s) {
    fprintf(stderr, "\r" "opn/cls: %zu/%zu  r/w: %zu/%zu (%llu/%llu)"
      " ioctl: %zu  bad: %zu",
      Stats.open, Stats.close, Stats.read, Stats.write, Stats.in,
      Stats.out, Stats.ioctl, Stats.badpack
    );
    if(Opt.verbose) { alarm(1); }
    return;
}

//-----------------------------------------------------------------------------
static int get_options(int *argc, const char ***argv) {
    static const char *_empty_argv[] = {NULL};
    char *tmp;
    struct poptOption options_table[] = {
        {NULL, 'D', POPT_ARG_STRING, &Opt.ovdev, 0,
         "Path to the RPL device", "file"},
        {NULL, 'O', POPT_ARG_STRING, &Opt.ofmt, 0,
         "Override OFMT variable", "string"},
        {NULL, 'Q', POPT_ARG_NONE, &Opt.nocreat, 0,
         "Debug: do not create any files", NULL},
        {NULL, 'U', POPT_ARG_STRING, &tmp, 'U',
         "User to change to", "user"},
        {NULL, 'c', POPT_ARG_STRING, &tmp, 'c',
         "Read configuration from file (on top of hardcoded conf)", "file"},
        {NULL, 's', POPT_ARG_NONE, &Opt.syslog, 0,
         "Use syslog rather than stderr for messages", NULL},
        {NULL, 'v', POPT_ARG_NONE, &Opt.verbose, 0,
         "Print statistics while rpld is running", NULL},
        POPT_AUTOHELP
        POPT_TABLEEND
    };

    poptContext ctx;
    const char **args;
    int c, argk;

    ctx = poptGetContext(**argv, *argc, *argv, options_table, 0);
    while((c = poptGetNextOpt(ctx)) >= 0) {
        switch(c) {
            case 'U':
                if(!pconfig_user(NULL, 0, tmp, NULL)) { return 0; }
                free(tmp);
                break;
            case 'c':
                read_config(tmp);
                free(tmp);
                break;
        }
    }

    if(c < -1) {
        fprintf(stderr, "%s: %s\n", poptBadOption(ctx, 0), poptStrerror(c));
        poptFreeContext(ctx);
        return 0;
    }

    if((args = poptGetArgs(ctx)) != NULL) {
        argk = SH_count_args(args);
        poptDupArgv(argk, args, argc, argv);
    } else {
        *argc = 0;
        *argv = _empty_argv;
    }

    poptFreeContext(ctx);
    return 1;
}

static int read_config(const char *file) {
    static struct rconfig_opt options_table[] = {
        {"OVDEVICE", RCONF_STRING, &Opt.ovdev,  NULL},
        {"FBDEVICE", RCONF_STRING, &Opt.fbdev,  NULL},
        {"OFMT",     RCONF_STRING, &Opt.ofmt,   NULL},
        {"USER",     RCONF_CB,     NULL,        pconfig_user},
        {NULL},
    };
    return HX_rconfig(file, options_table);
}

static int read_config_bp(const char *app_path, const char *file) {
    char *fpath = HX_strdup(app_path), *ptr, construct[MAXFNLEN];
    if((ptr = strrchr(fpath, '/')) == NULL) {
        construct[MAXFNLEN - 1] = '\0';
        strncpy(construct, file, MAXFNLEN - 1);
    } else {
        *ptr++ = '\0';
        snprintf(construct, MAXFNLEN, "%s/%s", fpath, file);
    }
    free(fpath);
    return read_config(construct);
}

static int pconfig_user(const char *key, unsigned char type, void *ptr,
 void *uptr) {
    if((Opt.user_id = getuidfromname(ptr)) < 0) {
        fprintf(stderr, "No such user: %s\n", (const char *)ptr);
        exit(EXIT_FAILURE);
    }
    return 1;
}

//-----------------------------------------------------------------------------
static char *getnamefromuid(uid_t uid, char *result, size_t len) {
    struct passwd ent, *ep;
    char additional[1024];
    getpwuid_r(uid, &ent, additional, 1024, &ep);
    if(ep == NULL) { return NULL; }
    strncpy(result, ep->pw_name, len - 1);
    result[len - 1] = '\0';
    return result;
}

static uid_t getuidfromname(const char *name) {
    struct passwd ent, *ep;
    char additional[1024];
    getpwnam_r(name, &ent, additional, 1024, &ep);
    if(ep == NULL) { return -1; }
    return ep->pw_uid;
}

static int notify(int lv, unsigned int opts, const char *fmt, ...) {
    if(Opt.verbose) {
        int rv;
        va_list argp;
        va_start(argp, fmt);
        if(opts & NF_NEWLINE) { fprintf(stderr, "\n"); }
        rv = vfprintf(stderr, fmt, argp);
        va_end(argp);
        return rv;
    } else if(Opt.syslog) {
        va_list argp;
        va_start(argp, fmt);
        vsyslog(lv, fmt, argp);
        va_end(argp);
        return 1;
    }
    return 0;
}

static char *replace(char *dest, const char *src, size_t count,
 char **catalog) {
    char *destp = dest;
    *destp = '\0';
    destp[count - 1] = '\0';

    while(*src != '\0' && count > 1) {
        size_t max = min_uint(strcspn(src, "%"), count - 1);
        char **catp = catalog, *key, *value;
        int got_spec = 0;

        strncpy(destp, src, max);
        src   += max;
        destp += max;
        count -= max;
        if(*src != '%' || count < 2) { break; }
        ++src;
        if(*src == '%') { // "%%"
            *destp++ = '%';
            if(--count < 2) { break; }
            ++src;
            continue;
        }
        if((unsigned char)*src >= 128 || !isalpha(*src)) {
            notify(LOG_WARNING, NF_NEWLINE, "%% without specifier in OFMT.\n");
            break;
        }

        while((key = *catp++) != NULL) {
            value = *catp++;
            if(*src == *key) {
                strncpy(destp, value, count - 1);
                max = strlen(destp);
                destp += max;
                count -= max;
                ++got_spec;
                break;
            }
        }
        if(count < 2) { break; }
        if(!got_spec) {
            notify(LOG_WARNING, NF_NEWLINE, "Invalid specifier: %%%c\n", *src);
        }
        ++src;
    }
    return dest;
}

static int value_cmp(void *pa, void *pb) {
    return (pa > pb) - (pa < pb);
}

//==[ End of file ]============================================================
