/*=============================================================================
ttyrpld - TTY replay daemon
user/shared.c - Shared data
  Copyright (C) Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/types.h>
#include <errno.h>
#include <sched.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "dev.h"
#include "rpl_packet.h"
#include "ushared.h"

//-----------------------------------------------------------------------------
void SH_check_balign(void) {
#define NALIGN(a, b) { \
    size_t diff = (void *)&(s.b) - (void *)&(s.a); \
    if(diff != sizeof(s.a)) { \
        fprintf(stderr, "Binary packet format alignment check failed:\n" \
         "a=%p b=%p diff=%d expect=%d\n", &(s.a), &(s.b), diff, sizeof(s.a)); \
    } \
  } 
    struct mem_packet s;
    NALIGN(dev, size);
    NALIGN(size, event);
    NALIGN(event, magic);
    NALIGN(magic, tv);
    return;
#undef NALIGN
}

int SH_count_args(const char **args) {
    int argk = 0;
    while(*args++ != NULL) { ++argk; }
    return argk;
}

ssize_t SH_read_wait(int fd, void *buf, size_t count) {
    /* A wrapper for read() which guarantees that all bytes requested will
    be in BUF after read_wait() returns. (Except if there is an error.)
    Note that it will retry to read when it hits EOF, so only use this on files
    which are still being written to! */
    size_t rem = count;

    while(rem > 0) {
        ssize_t eax = read(fd, buf, rem);
        if(eax < 0) { return -errno; }
        if(eax == rem) { break; }
        buf += eax;
        rem -= eax;
        sched_yield(); // usleep(1) could also be a possibility
    }
    return count;
}

off_t SH_skip(int fd, off_t offset, int do_wait) {
#define BUFSIZE 4096
    /* For files (and stuff) that can not be seeked in, use a slurping method
    to get to the wanted position. This only works for forward offsets. */
    off_t seekable = lseek(fd, 0, SEEK_CUR);
    size_t rem = offset;
    char buf[BUFSIZE];

    if(seekable != -1) { return lseek(fd, offset, SEEK_CUR); }

    if(do_wait) {
        while(rem > 0) {
            ssize_t eax = read(fd, buf, min_uint(BUFSIZE, rem));
            if(eax <= 0) { return -1; }
            if(eax == rem) { return 0; }
            rem -= eax;
            sched_yield();
        }
        return 0;
    }

    while(rem > 0) {
        ssize_t eax = read(fd, buf, min_uint(BUFSIZE, rem));
        if(eax < 0) { return -1; }
        if(eax == rem) { return 0; }
        rem -= eax;
        sched_yield();
    }

    return 0;
#undef BUFSIZE
}

void SH_swab(void *srcp, size_t count) {
    char *movp = srcp - 1;
    uint8_t x;

    while(count) {
        x = *++movp;
        movp[0] = movp[1];
        *++movp = x;
        count -= 2;
    }

    return;
}

//==[ End of file ]============================================================
