/*=============================================================================
ttyrpld - TTY replay daemon
user/global.c - Shared functions for user-space apps
  Copyright (C) Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/types.h>
#include <errno.h>
#include <pthread.h>
#include <sched.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "dev.h"
#include "global.h"

static int kver = 0;
static pthread_mutex_t kver_lock = PTHREAD_MUTEX_INITIALIZER;

//-----------------------------------------------------------------------------
int K_MINORBITS(void) {
    pthread_mutex_lock(&kver_lock);
    if(kver == 0) {
        char buf[64];
        FILE *fp;

        memset(&buf, '\0', 64);
        if((fp = fopen("/proc/version", "r")) != NULL) {
            fscanf(fp, "%*s %*s %60s", buf);
            if(strncmp(buf, "2.6", 3) == 0) { kver = 26; }
            else if(strncmp(buf, "2.4", 3) == 0) { kver = 24; }
            fclose(fp);
        }
    }
    pthread_mutex_unlock(&kver_lock);
    return (kver != 24) ? K26_MINORBITS : K24_MINORBITS;
}

int count_args(const char **args) {
    int argk = 0;
    while(*args++ != NULL) { ++argk; }
    return argk;
}

ssize_t read_wait(int fd, void *buf, size_t count) {
    /* A wrapper for read() which guarantees that all bytes requested will
    be in BUF after read_wait() returns. (Except if there is an error.)
    Note that it will retry to read when it hits EOF, so only use this on files
    which are still being written to! */
    size_t rem = count;

    while(rem > 0) {
        ssize_t eax = read(fd, buf, rem);
        if(eax < 0) { return -errno; }
        if(eax == rem) { break; }
        buf += eax;
        rem -= eax;
        sched_yield(); // usleep(1) could also be a possibility
    }
    return count;
}

off_t G_skip(int fd, off_t offset, int do_wait) {
    /* For files (and stuff) that can not be seeked in, use a slurping method
    to get to the wanted position. This only works for forward offsets. */
    off_t seekable = lseek(fd, 0, SEEK_CUR);
    size_t rem = offset;
    char buf[4096];

    if(seekable != -1) { return lseek(fd, offset, SEEK_CUR); }

    if(do_wait) {
        while(rem > 0) {
            ssize_t eax = read(fd, buf, min_uint(sizeof(buf), rem));
            if(eax <= 0) { return -1; }
            if(eax == rem) { return 0; }
            rem -= eax;
            sched_yield();
        }
        return 0;
    }

    while(rem > 0) {
        ssize_t eax = read(fd, buf, min_uint(sizeof(buf), rem));
        if(eax < 0) { return -1; }
        if(eax == rem) { return 0; }
        rem -= eax;
        sched_yield();
    }

    return 0;
}

void G_swab(void *srcp, size_t count) {
    char *movp = srcp - 1;
    uint8_t x;

    while(count) {
        x = *++movp;
        movp[0] = movp[1];
        *++movp = x;
        count -= 2;
    }

    return;
}

//==[ End of file ]============================================================
