/*=============================================================================
ttyrpld - TTY replay daemon
user/rplctl.c - INFOD client
  Copyright (C) Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/un.h>
#include <errno.h>
#include <popt.h>
#include <sched.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libHX.h>
#include "dev.h"
#include "global.h"
#include "rdsh.h"

// Function prototypes
static void mainloop(int);
static uint32_t getdev(const char *);
static int unix_client(const char *);

static int get_options(int *, const char ***);

// Global variables
static struct HXlld *Jobs = NULL;

//-----------------------------------------------------------------------------
int rplctl_main(int argc, const char **argv) {
    int fd;

    if((Jobs = HXlld_init(0)) == NULL) {
        perror("malloc()");
        return EXIT_FAILURE;
    }

    if(!get_options(&argc, &argv)) { return EXIT_FAILURE; }

    if((fd = unix_client(Opt.infod_port)) < 0) {
        fprintf(stderr, "connect() to %s failed: %s\n",
         Opt.infod_port, strerror(errno));
        return EXIT_FAILURE;
    }
    
    if(Jobs->itemcount == 0) {
        if(argc < 1) {
            /* If no arguments are given at all, get information about all
            ttys and INFOD. */
            HXlld_push(Jobs, (void *)IFP_GETINFO);
            HXlld_push(Jobs, NULL);
        } else {
            while(*argv != NULL) {
                uint32_t dev;
                if((dev = getdev(*argv)) != 0) {
                    HXlld_push(Jobs, (void *)IFP_GETINFO);
                    HXlld_push(Jobs, (void *)dev);
                }
                ++argv;
            }
        }
    }

    mainloop(fd);
    close(fd);
    HXlld_free(Jobs);
    return EXIT_SUCCESS;
}

static void mainloop(int fd) {
    unsigned char req;
    char buf[4096];

    while((req = (unsigned char)(int)HXlld_shift(Jobs)) != 0) {
        /* Protocol:
        1. rplctl sends request type (1 byte)
        (1a. infod connection ends if request is IFP_NONE)
        2. rplctl sends device number
        3. infod sends size of reply
        4. rplctl stops if size is zero, otherwise displays reply
           jump to 3
        */
        uint32_t dev = (uint32_t)HXlld_shift(Jobs);

        send_wait(fd, &req, sizeof(unsigned char), 0);
        PP_swab(&dev, sizeof(uint32_t));
        send_wait(fd, &dev, sizeof(uint32_t), 0);

        while(1) {
            uint32_t reply_size = 0;
            fd_set rd;

            recv(fd, &reply_size, sizeof(uint32_t), MSG_WAITALL);
            PP_swab(&reply_size, sizeof(uint32_t));
            if(reply_size == 0) { break; }

            while(reply_size > 0) {
                FD_ZERO(&rd);
                FD_SET(fd, &rd);
                select(fd + 1, &rd, NULL, NULL, NULL);
                if(FD_ISSET(fd, &rd)) {
                    ssize_t eax = recv(fd, buf,
                     min_uint(sizeof(buf), reply_size), 0);
                    write(STDOUT_FILENO, buf, eax);
                    reply_size -= eax;
                }
            }
        }
    }

    req = IFP_NONE;
    send_wait(fd, &req, sizeof(unsigned char), 0);
    return;
}

//-----------------------------------------------------------------------------
static uint32_t getdev(const char *s) {
    uint32_t major, minor;
    int found = 0;

    if(s == NULL) { return 0; }
    if(*s == '=') {
        /* The user can specify a major-minor number combination in the form of
        "=MAJOR,MINOR" (BNF: "=" MAJOR "," MINOR)... */
        char *sdup = HX_strdup(s), *p = sdup, *minor_str;
        if((minor_str = strpbrk(p + 1, ",.:")) == NULL || p + 1 == minor_str) {
            fprintf(stderr, "getdev(): Incorrect device number specification"
             " \"%s\", should be \"=%%d:%%d\"\n", s);
            free(sdup);
            return 0;
        }
        *minor_str++ = '\0';
        major = strtoul(++p, NULL, 0);
        minor = strtoul(minor_str, NULL, 0);
        free(sdup);
    } else {
        /* ... or a pathname specification to a device node whose rdev number
        will be extracted. Note that we do not check for S_ISCHR or S_ISBLK,
        so passing /dev/hda here will effectively mean /dev/ttyp0. */
        const char *dirs[] = {".", "/dev", NULL}, **dirp = dirs;
        struct stat sb;

        while(*dirp != NULL) {
            char buf[MAXFNLEN];
            snprintf(buf, MAXFNLEN, "%s/%s", *dirp, s);
            if(stat(buf, &sb) == 0) { ++found; break; }
            if(errno != ENOENT) {
                fprintf(stderr, "Cannot stat %s: %s\n", buf, strerror(errno));
                return 0;
            }
            ++dirp;
        }
        if(!found) {
            fprintf(stderr, "Cannot find %s\n", s);
            return 0;
        }

        major = GLIBC_MAJOR(sb.st_rdev);
        minor = GLIBC_MINOR(sb.st_rdev);
    }
    return K26_MKDEV(major, minor);
}

static int unix_client(const char *port) {
    struct sockaddr_un sk;
    int fd = -1;

    sk.sun_family = AF_UNIX;
    strncpy(sk.sun_path, port, sizeof(sk.sun_path));
    sk.sun_path[sizeof(sk.sun_path) - 1] = '\0';

    if((fd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0 ||
     connect(fd, (struct sockaddr *)&sk, sizeof(struct sockaddr_un)) < 0) {
        return -1;
    }

    return fd;
}

//-----------------------------------------------------------------------------
static int get_options(int *argc, const char ***argv) {
    static const char *_empty_argv[] = {NULL};
    char *tmp;
    struct poptOption options_table[] = {
        {NULL, 'A', POPT_ARG_STRING, &tmp, 'A',
         "Capture data from tty", "tty"},
        {NULL, 'D', POPT_ARG_STRING, &tmp, 'D',
         "Ignore data from tty until next tty session", "tty"},
        {NULL, 'F', POPT_ARG_NONE, NULL, 'F',
         "Flush tty table", NULL},
        {NULL, 'J', POPT_ARG_STRING, &tmp, 'J',
         "Ignore data from tty", "tty"},
        {NULL, 'X', POPT_ARG_STRING, &tmp, 'X',
         "Call log_close() for the given tty", "tty"},
        {NULL, 'f', POPT_ARG_STRING, &Opt.infod_port, 0,
         "Path to the INFOD socket", "file"},
        POPT_AUTOHELP
        POPT_TABLEEND
    };

    poptContext ctx;
    const char **args;
    int c, argk;

    ctx = poptGetContext(**argv, *argc, *argv, options_table, 0);
    while((c = poptGetNextOpt(ctx)) >= 0) {
        static const int mapping[] = {
            ['A'] = IFP_ACTIVATE,
            ['D'] = IFP_DEACTIVSES,
            ['J'] = IFP_DEACTIVATE,
            ['X'] = IFP_REMOVE,
        };
        if(c == 'A' || c == 'D' || c == 'J' || c == 'X') {
            uint32_t dev;
            if((dev = getdev(tmp)) != 0) {
                HXlld_push(Jobs, (void *)mapping[c]);
                HXlld_push(Jobs, (void *)dev);
            }
            free(tmp);
        } else if(c == 'F') {
            HXlld_push(Jobs, (void *)IFP_FLUSH);
            HXlld_push(Jobs, (void *)NULL);
        }
    }

    if(c < -1) {
        fprintf(stderr, "%s: %s\n", poptBadOption(ctx, 0), poptStrerror(c));
        poptFreeContext(ctx);
        return 0;
    }

    if((args = poptGetArgs(ctx)) != NULL) {
        argk = count_args(args);
        poptDupArgv(argk, args, argc, argv);
    } else {
        *argc = 0;
        *argv = _empty_argv;
    }

    poptFreeContext(ctx);
    return 1;
}

//==[ End of file ]============================================================
