/*=============================================================================
ttyrpld - TTY replay daemon
user/rpld.c - User space daemon (filtering, etc.)
  Copyright (C) Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/mman.h> // mlock()
#include <sys/stat.h>
#include <sys/time.h> // gettimeofday()
#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <popt.h>
#include <pthread.h>
#include <pwd.h>
#include <signal.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
extern int posix_memalign(void **, size_t, size_t);
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include <libHX.h>
#include "dev.h"
#include "rpl_ioctl.h"
#include "rpl_packet.h"
#include "global.h"
#include "rdsh.h"
#define NODEPATH "/var/run/.rpldev"

// The prototype order is the same as the function body order
static void mainloop(int);

static int packet_action1(struct mem_packet *);
static int packet_action2(struct mem_packet *, struct tty *);
static int packet_action3(struct mem_packet *, struct tty *, int);

static void log_cuid(struct tty *);
static int log_open(struct tty *);
static void log_write(struct mem_packet *, struct tty *, int);

static int init_device(const char *);
static int init_sbuffer(int);
static void deinit_sbuffer(void);
static const char *mknod_rpldev(void);
static int init_sighandler(void);
static void sighandler_int(int);
static void sighandler_alrm(int);
static void sighandler_pipe(int);

static int get_options(int *, const char ***);
static int read_config(const char *);
static int read_config_bp(const char *, const char *);
static int pconfig_user(const char *, unsigned char, void *, void *);

inline static const char *basename(const char *);
static char *getnamefromuid(uid_t, char *, size_t);
static uid_t getuidfromname(const char *);
static int value_cmp(void *, void *);

// Global variables (begin with uppercase character)
static void *Buffer = NULL;
static struct {
    int _running, nocreat;
    char *ovdev, *fbdev, *ofmt;
    long bsize;

    int infod_start;
} Opt = {
    ._running    = 1,
    .bsize       = 16384,
    .ovdev       = NULL,          // -D
    .fbdev       = "/dev/rpl:/dev/misc/rpl",
    .nocreat     = 0,             // -Q
    .ofmt        = "%u@%d.%t.%l", // -O
    .infod_start = 0,             // -I
};

//-----------------------------------------------------------------------------
int main(int argc, const char **argv) {
    pthread_t infod_id;
    int fd;

    /* Yep, the config file is what is needed by all three
    (rpld, infod, rplctl). */
    if(!read_config_bp(*argv, "rpld.conf")) {
        return EXIT_FAILURE;
    }

    if(strcmp(basename(*argv), "rplctl") == 0) {
        return rplctl_main(argc, argv);
    }

    if(!get_options(&argc, &argv)) { return EXIT_FAILURE; }
    memset(&Stats, 0, sizeof(Stats));

    if(GOpt.verbose) {
        printf(
          "# rpld " TTYRPLD_VERSION "\n"
          "This program comes with ABSOLUTELY NO WARRANTY; it is free"
          " software\n"
          "and you are welcome to redistribute it under certain conditions;\n"
          "for details see the doc/GPL2.txt file which should have come with\n"
          "this program.\n\n"
        );
    }

    if((Ttys = HXbtree_init(HXBT_ASSOC | HXBT_LKEY | HXBT_XCPY | HXBT_FCMP,
     value_cmp)) == NULL) {
        perror("HXbtree_init()");
        return EXIT_FAILURE;
    }

    if((fd = init_device((Opt.ovdev != NULL) ? Opt.ovdev : Opt.fbdev)) < 0) {
        fprintf(stderr, "No device could be opened, aborting.\n");
        return EXIT_FAILURE;
    }

    if(init_sbuffer(fd) < 0) { return EXIT_FAILURE; }
    init_sighandler();

    // Callback tick for statistics
    if(GOpt.verbose) { alarm(1); }

    if(Opt.infod_start) { infod_init(); }

    if(GOpt.syslog) { openlog("rpld", LOG_PID, LOG_DAEMON); }

    // Not seteuid(), because it shall not be posibble to regain privs.
    if(GOpt.user_id >= 0) { setuid(GOpt.user_id); }

    if(Opt.infod_start) { pthread_create(&infod_id, NULL, infod_main, NULL); }

    mainloop(fd);

    if(Opt.infod_start) {
        unlink(GOpt.infod_port);
        pthread_cancel(infod_id);
        pthread_join(infod_id, NULL);
    }

    close(fd);
    deinit_sbuffer();
    return EXIT_SUCCESS;
}

static void mainloop(int fd) {
    while(Opt._running) {
        struct mem_packet packet;
        struct tty *tty;

        if(read(fd, &packet, sizeof(struct rpld_packet)) <
         sizeof(struct rpld_packet)) {
            Opt._running = 0;
            break;
        }

        PP_swab(&packet.dev, sizeof(packet.dev));
        PP_swab(&packet.size, sizeof(packet.size));

        if(packet.magic == MAGIC_2_6) {
            // nothing needed
        } else if(packet.magic == MAGIC_2_4) {
            packet.magic = MAGIC_2_6;
            packet.dev   = K26_MKDEV(K24_MAJOR(packet.dev),
             K24_MINOR(packet.dev));
        } else {
            ++Stats.badpack;
            notify(LOG_WARNING, NF_NEWLINE,
             "Bogus packet (magic is 0x%X)!\n", packet.magic);
            continue;
        }

        /* Timestamp is taken here instead of in the Kernel module to reduce
        the amount of data which has to be copied over the device. */
        gettimeofday(&packet.tv, NULL);

        if(!packet_action1(&packet)) {
            G_skip(fd, packet.size, 0);
            continue;
        }

        pthread_mutex_lock(&Ttys_lock);
        tty = get_tty(packet.dev);
        pthread_mutex_unlock(&Ttys_lock);

        if(tty == NULL || !packet_action2(&packet, tty)) {
            G_skip(fd, packet.size, 0);
            continue;
        }

        packet_action3(&packet, tty, fd);
    }

    return;
}

//-----------------------------------------------------------------------------
static int packet_action1(struct mem_packet *packet) {
    /* pty masters are an exact copy of their slaves,
    so no need to record them too. */
    register long maj = K26_MAJOR(packet->dev);

    if(maj >= 128 && maj < 136 /* Unix98 ptm (-) */) {
        if(packet->event != EV_CLOSE) { return 0; }
        /* When closing an xterm window or SSH session, the EV_CLOSE event
        actually happens on the master side. To keep it easy, it's device
        signature is simply changed to the slave. */
        packet->dev = K26_MKDEV(K26_MAJOR(packet->dev) + 8,
         K26_MINOR(packet->dev));
    } else if(maj == 2 /* BSD ptm (/dev/ptypXY) */) {
        if(packet->event != EV_CLOSE) { return 0; }
        packet->dev = K26_MKDEV(K26_MAJOR(3), K26_MINOR(packet->dev));
    }

    /* The excluded ptm packets above are not counted at all, whereas
    "ignored" packets are. See below for ignore rules. */

    switch(packet->event) {
        case EV_OPEN:
            ++Stats.open;
            break;
        case EV_READ:
            ++Stats.read;
            Stats.in += packet->size;
            break;
        case EV_WRITE:
            ++Stats.write;
            Stats.out += packet->size;
            break;
        case EV_IOCTL:
            ++Stats.ioctl;
            break;
        case EV_CLOSE:
            ++Stats.close;
            break;
        default:
            notify(LOG_WARNING, NF_NEWLINE, "Unknown packet type 0x%X\n",
             packet->event);
            return 0;
    }

    if(packet->event == EV_IOCTL) {
        /* ioctl requests are actually logged by the rpldev Kernel module,
        and I do see a use for them. But: it's simply not
        implemented yet ;-) */
        return 0;
    }

    return 1;
}

static int packet_action2(struct mem_packet *packet, struct tty *tty) {
    /* The name is wrong when you look at the caller, since postaction()
    actually comes before packet_action(). Anyway, it's post the get_tty(). */
    if(tty->status != IFP_ACTIVATE) {
        return 0;
    }
    switch(packet->event) {
        case EV_READ:
            tty->in += packet->size;
            break;
        case EV_WRITE:
            tty->out += packet->size;
            break;
    }
    return 1;
}

static int packet_action3(struct mem_packet *packet, struct tty *tty, int fd) {
    switch(packet->event) {
        case EV_OPEN:
            /* EV_OPEN is not handled since a logfile will automatically be opened
            in log_write() if necessary. */
            log_cuid(tty);
            break;
        case EV_READ:
        case EV_WRITE:
            log_write(packet, tty, fd);
            break;
        case EV_CLOSE:
            log_close(tty);
            break;
    }
    return 1;
}

//-----------------------------------------------------------------------------
static void log_cuid(struct tty *tty) {
    struct stat sb;
    char buf[32];

    if(tty->uid == -1) { return; }
    G_devname_fs(tty->dev, buf, sizeof(buf));
    if(stat(buf, &sb) == 0 && sb.st_uid != tty->uid) {
        tty->in = tty->out = 0;
        log_open(tty);
    }
    return;
}

static int log_open(struct tty *tty) {
    char buf[MAXFNLEN], devnode[32], fmday[16], fmtime[16], user[32];
    struct HX_repmap catalog[] = {{'d', NULL, fmday}, {'l', NULL, devnode},
     {'t', NULL, fmtime}, {'u', NULL, user}, {0}};

    { // User
      struct stat sb;
      G_devname_fs(tty->dev, devnode, sizeof(devnode));
      if(stat(devnode, &sb) != 0) {
          strcpy(user, "NONE");
      } else {
          tty->uid = sb.st_uid;
          if(getnamefromuid(sb.st_uid, user, sizeof(user)) == NULL) {
              snprintf(user, sizeof(user), "%d", sb.st_uid);
          }
      }
    }

    { // Time
      time_t now = time(NULL);
      struct tm now_tm;
      localtime_r(&now, &now_tm);
      strftime(fmday,  sizeof(fmday),  "%Y%m%d", &now_tm);
      strftime(fmtime, sizeof(fmtime), "%H%M%S", &now_tm);
    }

    // Simple tty name
    G_devname_nm(tty->dev, devnode, sizeof(devnode));
    HX_strrep(buf, MAXFNLEN, Opt.ofmt, catalog);

    // Open log
    HX_strclone(&tty->file, basename(buf));
    if(Opt.nocreat) { snprintf(buf, MAXFNLEN, "/dev/null"); }
    if((tty->fd = open(buf, O_WRONLY | O_CREAT | O_APPEND, 0600)) < 0) {
        notify(LOG_ERR, NF_NEWLINE, "Could not open %s: %s\n",
         buf, strerror(errno));
    }

    // ident header (optional)
    if(tty->fd >= 0) {
        struct disk_packet p = {
            .event = EV_IDENT,
            .magic = MAGIC_SIG,
            .tv    = {-1, -1},
        };
        size_t s;

        buf[MAXFNLEN - 1] = '\0';
        strncpy(buf, "ttyrpld " TTYRPLD_VERSION, MAXFNLEN - 1);
        s = p.size = strlen(buf);
        PP_swab(&p.size, sizeof(p.size));
        write(tty->fd, &p, sizeof(struct disk_packet));
        write(tty->fd, buf, s);
    }

    return tty->fd;
}

static void log_write(struct mem_packet *packet, struct tty *tty, int fd) {
    /* Slap a timestamp around the packet and write it out to disk. Open the
    log if necessary. */
    size_t s = packet->size;
    int read_left;

    if(tty->fd == -1) { log_open(tty); }
    PP_swab(&packet->size, sizeof(packet->size));
    PP_swab(&packet->tv.tv_sec, sizeof(packet->tv.tv_sec));
    PP_swab(&packet->tv.tv_usec, sizeof(packet->tv.tv_usec));
    write(tty->fd, &packet->size, sizeof(struct disk_packet));
    read_left = s;

    while(read_left > 0) { // XXX replace with read_through
        int have_read = read(fd, Buffer, min_uint(read_left, Opt.bsize));
        write(tty->fd, Buffer, have_read);
        read_left -= have_read;
    }

    return;
}

//-----------------------------------------------------------------------------
static int init_device(const char *in_devs) {
    const char *new_dev = NULL;
    int fd = -1;

    if(Opt.ovdev == NULL && (new_dev = mknod_rpldev()) != NULL) {
        if((fd = open(new_dev, O_RDONLY)) == -1) {
            new_dev = NULL;
            perror("Could not connect to RPL device");
        } else if(GOpt.verbose) {
            printf("Connected to RPL device\n");
        }
        unlink(new_dev);
    }

    if(new_dev == NULL) {
        /* No, this will not change into an else if, because new_dev might be
        set to something else above! */
        char *copy = HX_strdup(in_devs), *devs = copy, *devp;

        while((devp = HX_strsep(&devs, ":")) != NULL) {
            if((fd = open(devp, O_RDONLY)) >= 0) {
                if(GOpt.verbose) { printf("Connected to %s\n", devp); }
                break;
            }
            if(errno != ENOENT) {
                fprintf(stderr, "Could not open %s even though it exists: %s\n"
                 "Trying next device...\n", devp, strerror(errno));
            }
            ++devp;
        }
        free(copy);
    }

    return fd;
}

static int init_sbuffer(int fd) {
    int eax;

    if((eax = ioctl(fd, RPL_IOC_GETBUFSIZE, 0)) > 0) {
        Opt.bsize = eax;
    }

    if(posix_memalign(&Buffer, sysconf(_SC_PAGESIZE), Opt.bsize) != 0) {
        eax = errno;
        perror("malloc()/posix_memalign()");
        return -(errno = eax);
    }

    /* Done so that this memory area can not be swapped out, which is essential
    against buffer overruns. */
    mlock(Buffer, Opt.bsize);
    return 1;
}

static void deinit_sbuffer(void) {
    munlock(Buffer, Opt.bsize);
    free(Buffer);
    return;
}

static const char *mknod_rpldev(void) {
    int32_t minor = -1;
    char buf[128];
    FILE *fp;

    if((fp = fopen("/proc/misc", "r")) == NULL) {
        perror("Could not open /proc/misc");
        return NULL;
    }

    while(fgets(buf, sizeof(buf), fp) != NULL) {
        char *ptr = buf, *name;
        while(!isdigit(*ptr)) { ++ptr; }
        minor = strtol(ptr, NULL, 0);
        while(isdigit(*ptr)) { ++ptr; }
        while(isspace(*ptr)) { ++ptr; }

        name = ptr;
        while(isalnum(*ptr)) { ++ptr; }
        *ptr = '\0';
        if(strcmp(name, "rpl") == 0) { break; }
        minor = -1;
    }

    fclose(fp);

    if(minor == -1) {
        printf("RPL module does not seem to be loaded\n");
        return NULL;
    }

    if(mknod(NODEPATH, S_IFCHR | 0400, GLIBC_MKDEV(10, minor)) != 0) {
        return NULL;
    }

    chown(NODEPATH, GOpt.user_id, 0);
    return NODEPATH;
}

static int init_sighandler(void) {
    struct sigaction s_int, s_alrm, s_pipe;

    s_int.sa_handler = sighandler_int;
    s_int.sa_flags   = SA_RESTART;
    sigemptyset(&s_int.sa_mask);

    s_alrm.sa_handler = sighandler_alrm;
    s_alrm.sa_flags   = SA_RESTART;
    sigemptyset(&s_alrm.sa_mask);

    s_pipe.sa_handler = sighandler_pipe;
    s_pipe.sa_flags   = SA_RESTART;
    sigemptyset(&s_pipe.sa_mask);

    /* All sigactions() shall be executed, however, if one fails, this function
    shall return <= 0, otherwise >0 upon success.
    Geesh, I love these constructs. */
    return !(!!sigaction(SIGINT, &s_int, NULL) + 
     !!sigaction(SIGTERM, &s_int, NULL) +
     !!sigaction(SIGALRM, &s_alrm, NULL) +
     !!sigaction(SIGPIPE, &s_pipe, NULL));
}

static void sighandler_int(int s) {
    if(Opt._running-- == 0) {
        if(GOpt.verbose) {
            printf("Second time we received SIGINT/SIGTERM,"
             " exiting immediately.\n");
        }
        exit(EXIT_FAILURE);
    }
    if(GOpt.verbose) {
        printf("\n" "Received SIGINT/SIGTERM, shutting down.\n");
    } else {
        // If this is not done, process hangs until some fs activity
        printf("\n");
    }
    Opt._running = 0;
    return;
}

static void sighandler_alrm(int s) {
    fprintf(stderr, "\r\e[2K" "opn/cls: %zu/%zu  r/w: %zu/%zu (%llu/%llu)"
      " ioctl: %zu  bad: %zu",
      Stats.open, Stats.close, Stats.read, Stats.write, Stats.in,
      Stats.out, Stats.ioctl, Stats.badpack
    );
    if(GOpt.verbose) { alarm(1); }
    return;
}

static void sighandler_pipe(int s) {
    fprintf(stderr, "\n" "[%d] Received SIGPIPE\n", getpid());
    return;
}

//-----------------------------------------------------------------------------
static int get_options(int *argc, const char ***argv) {
    static const char *_empty_argv[] = {NULL};
    char *tmp;
    struct poptOption options_table[] = {
        {NULL, 'D', POPT_ARG_STRING, &Opt.ovdev, 0,
         "Path to the RPL device", "file"},
        {NULL, 'I', POPT_ARG_NONE, &Opt.infod_start, 0,
         "Start INFOD subcomponent", NULL},
        {NULL, 'O', POPT_ARG_STRING, &Opt.ofmt, 0,
         "Override OFMT variable", "string"},
        {NULL, 'Q', POPT_ARG_NONE, &Opt.nocreat, 0,
         "Debug: do not create any files", NULL},
        {NULL, 'U', POPT_ARG_STRING, &tmp, 'U',
         "User to change to", "user"},
        {NULL, 'c', POPT_ARG_STRING, &tmp, 'c',
         "Read configuration from file (on top of hardcoded conf)", "file"},
        {NULL, 's', POPT_ARG_NONE, &GOpt.syslog, 0,
         "Print warnings/errors to syslog", NULL},
        {NULL, 'v', POPT_ARG_NONE, &GOpt.verbose, 0,
         "Print statistics while rpld is running", NULL},
        POPT_AUTOHELP
        POPT_TABLEEND
    };

    poptContext ctx;
    const char **args;
    int c, argk;

    ctx = poptGetContext(**argv, *argc, *argv, options_table, 0);
    while((c = poptGetNextOpt(ctx)) >= 0) {
        switch(c) {
            case 'U':
                if(!pconfig_user(NULL, 0, tmp, NULL)) { return 0; }
                free(tmp);
                break;
            case 'c':
                read_config(tmp);
                free(tmp);
                break;
        }
    }

    if(c < -1) {
        fprintf(stderr, "%s: %s\n", poptBadOption(ctx, 0), poptStrerror(c));
        poptFreeContext(ctx);
        return 0;
    }

    if((args = poptGetArgs(ctx)) != NULL) {
        argk = count_args(args);
        poptDupArgv(argk, args, argc, argv);
    } else {
        *argc = 0;
        *argv = _empty_argv;
    }

    poptFreeContext(ctx);
    return 1;
}

static int read_config(const char *file) {
    static struct rconfig_opt options_table[] = {
        {"OVDEVICE", RCONF_STRING, &Opt.ovdev, NULL},
        {"FBDEVICE", RCONF_STRING, &Opt.fbdev, NULL},
        {"OFMT",     RCONF_STRING, &Opt.ofmt,  NULL},
        {"USER",     RCONF_CB,     NULL,       pconfig_user},

        {"INFOD_PORT", RCONF_STRING, &GOpt.infod_port, NULL},
        {"START_INFOD", RCONF_IBOOL, &Opt.infod_start, NULL},
        {NULL},
    };
    return HX_rconfig(file, options_table);
}

static int read_config_bp(const char *app_path, const char *file) {
    char *fpath = HX_strdup(app_path), *ptr, construct[MAXFNLEN];
    if((ptr = strrchr(fpath, '/')) == NULL) {
        construct[MAXFNLEN - 1] = '\0';
        strncpy(construct, file, MAXFNLEN - 1);
    } else {
        *ptr++ = '\0';
        snprintf(construct, MAXFNLEN, "%s/%s", fpath, file);
    }
    free(fpath);
    return read_config(construct);
}

static int pconfig_user(const char *key, unsigned char type, void *ptr,
 void *uptr) {
    if((GOpt.user_id = getuidfromname(ptr)) < 0) {
        fprintf(stderr, "No such user: %s\n", (const char *)ptr);
        exit(EXIT_FAILURE);
    }
    return 1;
}

//-----------------------------------------------------------------------------
inline static const char *basename(const char *s) {
    // Return file component of a full-path filename
    const char *p = strrchr(s, '/');
    if(p != NULL) { return p + 1; }
    return s;
}

static char *getnamefromuid(uid_t uid, char *result, size_t len) {
    struct passwd ent, *ep;
    char additional[1024];
    getpwuid_r(uid, &ent, additional, sizeof(additional), &ep);
    if(ep == NULL) { return NULL; }
    strncpy(result, ep->pw_name, len - 1);
    result[len - 1] = '\0';
    return result;
}

static uid_t getuidfromname(const char *name) {
    struct passwd ent, *ep;
    char additional[1024];
    getpwnam_r(name, &ent, additional, sizeof(additional), &ep);
    if(ep == NULL) { return -1; }
    return ep->pw_uid;
}

static int value_cmp(void *pa, void *pb) {
    return (pa > pb) - (pa < pb);
}

//==[ End of file ]============================================================
