/*=============================================================================
ttyrpld - TTY replay daemon
user/rdsh.c - Shared functions for RPLD/INFOD/RPLCTL
  Copyright (C) Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/socket.h>
#include <sys/types.h>
#include <errno.h>
#include <pthread.h>
#include <sched.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>

#include <libHX.h>
#include "dev.h"
#include "global.h"
#include "rdsh.h"

// Functions
inline static int K_MINORBITS(void);
static void setup_kminorbits(void);

// External Variables
pthread_mutex_t Ttys_lock = PTHREAD_MUTEX_INITIALIZER;
struct HXbtree *Ttys = NULL;
struct Statmap_t Stats;
struct GOptmap_t GOpt = {
    .user_id    = -1,
    .syslog     = 0,
    .verbose    = 0,
    .infod_port = "/var/run/.rplinfod_socket",
};

// Variables
static unsigned char kver = 0;
static pthread_mutex_t kver_lock = PTHREAD_MUTEX_INITIALIZER;

//-----------------------------------------------------------------------------
static const char *m_idx = "pqrstuvwxyzabcdef", *s_idx = "0123456789abcdef";

char *G_devname_nm(uint32_t dev, char *buf, size_t count) {
    /* This function returns a "simple name" which can be integrated into
    rpld's log file name scheme. */
    register unsigned long major = K26_MAJOR(dev), minor = K26_MINOR(dev);

    switch(major) {
        case 2: // BSD pty masters
            snprintf(buf, count, "pty%c%c",
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 3: // BSD pty slaves
            snprintf(buf, count, "tty%c%c",
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 4:
            // Virtual console (0-63) and serial console (64-255)
            if(minor < 64) { snprintf(buf, count, "vc-%lu", minor); }
            else { snprintf(buf, count, "ttyS%lu", minor - 64); }
            break;
        case 5:
            if(minor == 1) { snprintf(buf, count, "console"); }
            else { goto unknown; }
        case 128 ... 135:
            // Unix98 pty masters (got no real /dev entry, though, see below)
            snprintf(buf, count, "ptm-%lu",
             ((major - 128) << K_MINORBITS()) | minor);
            break;
        case 136 ... 143: // Unix98 pty slaves
            /* Surprise, surprise. In 2.4, Unix98 ptys use major 136..143 while
            in 2.6 only 136 thanks to the bigger minor numbers. Anyway, can't
            be bad to have support for 136..143 for 2.6 ;-) */
            snprintf(buf, count, "pts-%lu",
             ((major - 136) << K_MINORBITS()) | minor);
            /* Thus: 2.4(137,2) == 2.6(136, 258)
               And: 2.6(137,2) == 2.6(136, 1048578)... kinda
            Same weirdance goes for pty masters. Lucky BSD, ever since only
            got 256 devices fitting in one major, and still at it :-D */
            break;
        default:
            goto unknown;
    }
    return buf;

unknown:
    /* Not every char device is a tty, and I even do not want to include a
    name database for all possible tty devices. */
    snprintf(buf, count, "%lu-%lu", K26_MAJOR(dev), minor);
    return buf;
}

char *G_devname_fs(uint32_t dev, char *buf, size_t count) {
    /* This function returns a device filename generated from major/minor
    number, which is then used for stat(). */
    register unsigned long major = K26_MAJOR(dev), minor = K26_MINOR(dev);

    switch(K26_MAJOR(dev)) {
        case 2: // BSD pty masters
            snprintf(buf, count, "/dev/pty%c%c", // devfs: /dev/pty/m%d
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 3: // BSD pty slaves
            snprintf(buf, count, "/dev/tty%c%c", // devfs: /dev/pty/s%d
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 4:
            // Virtual console (0-63) and serial console (64-255)
            if(minor < 64) { snprintf(buf, count, "/dev/tty%lu", minor); }
            else { snprintf(buf, count, "/dev/ttyS%lu", minor - 64); }
            break;
        case 5:
            if(minor == 1) { snprintf(buf, count, "/dev/console"); }
            else { return NULL; }
        case 128 ... 135:
            /* Unix98 pty masters. They usually have no device nodes in /dev,
            and should instead be accessed over the multiplexer /dev/ptmx.
            However IIRC, devfs popped them up in /dev/ptm. */
            snprintf(buf, count, "/dev/ptm/%lu",
             ((major - 128) << K_MINORBITS()) | minor);
            break;
        case 136 ... 143: // Unix98 pty slaves
            snprintf(buf, count, "/dev/pts/%lu",
             ((major - 136) << K_MINORBITS()) | minor);
            break;
        default:
            return NULL;
    }
    return buf;
}

struct tty *get_tty(uint32_t dev) {
    struct HXbtree_node *ts;
    struct tty *ret = NULL, *tty;

    if((ret = HXbtree_get(Ttys, (void *)dev)) != NULL) { return ret; }
    if((tty = malloc(sizeof(struct tty))) == NULL) { return NULL; }

    tty->dev    = dev;
    tty->uid    = -1;
    tty->fd     = -1;
    tty->status = IFP_ACTIVATE;
    tty->in     = tty->out = 0;
    tty->file   = NULL;

    if((ts = HXbtree_add(Ttys, (void *)dev, tty)) == NULL) {
        free(tty);
        notify(LOG_ERR, NF_NEWLINE, "Memory allocation failure");
        return NULL;
    }

    return ts->data;
}

void log_close(struct tty *tty) {
    /* Close the logfile and release the tty struct if it does not have special
    options are set. */
    pthread_mutex_lock(&Ttys_lock);

    close(tty->fd);
    tty->fd = -1;
    if(tty->file != NULL) {
        free(tty->file);
        tty->file = NULL; // infod
    }

    if(tty->status != IFP_DEACTIVATE) {
        /* If the status is IFP_ACTIVATED, it is reinstantiated upon next
        get_tty(). If it is IFP_DEACTIVSES, it will change to IFP_ACTIVATED,
        as per definition. Only keep IFP_DEACTIVATED. */
        HXbtree_del(Ttys, (void *)tty->dev);
        free(tty);
    }
    pthread_mutex_unlock(&Ttys_lock);
    return;
}

int notify(int lv, unsigned int opts, const char *fmt, ...) {
    if(GOpt.verbose) {
        int rv;
        va_list argp;
        va_start(argp, fmt);
        if(opts & NF_NEWLINE) { fprintf(stderr, "\n"); }
        rv = vfprintf(stderr, fmt, argp);
        va_end(argp);
        return rv;
    }
    if(GOpt.syslog) {
        va_list argp;
        va_start(argp, fmt);
        vsyslog(lv, fmt, argp);
        va_end(argp);
        return 1;
    }
    return 0;
}

ssize_t send_wait(int fd, const void *buf, size_t count, int flags) {
    size_t rem = count;
    while(rem > 0) {
        ssize_t eax = send(fd, buf, rem, flags);
        if(eax < 0) { return -errno; }
        if(eax == rem) { break; }
        buf += eax;
        rem -= eax;
        sched_yield();
    }
    return count;
}

//-----------------------------------------------------------------------------
inline static int K_MINORBITS(void) {
    pthread_mutex_lock(&kver_lock);
    if(kver == 0) { setup_kminorbits(); }
    pthread_mutex_unlock(&kver_lock);
    return (kver != 24) ? K26_MINORBITS : K24_MINORBITS;
}

static void setup_kminorbits(void) {
    char buf[64];
    FILE *fp;

    memset(&buf, '\0', 64);
    if((fp = fopen("/proc/version", "r")) == NULL) { return; }

    fscanf(fp, "%*s %*s %60s", buf);
    if(strncmp(buf, "2.6", 3) == 0) { kver = 26; }
    else if(strncmp(buf, "2.4", 3) == 0) { kver = 24; }
    fclose(fp);
    return;
}

//==[ End of file ]============================================================
