/*=============================================================================
ttyrpld - TTY replay daemon
user/infod.c - INFO socket for RPLD
  Copyright © Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/un.h>
#include <pthread.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include <libHX.h>
#include "dev.h"
#include "global.h"
#include "rdsh.h"

// Function prototypes
static void *client_thread(void *);
static void flush_table(void);
static void set_session_status(uint32_t, int);
static void getinfo_text(uint32_t, int);
static void getinfo(uint32_t, int);
static void zero_counters(void);

static void block_signals(void);
static void print_entry_text(int, struct tty *);
static void print_entry(int, struct tty *);
static int skprintf(int, const char *, ...);
static int unix_server(const char *);

// Variables
static int Svfd = -1;

//-----------------------------------------------------------------------------
void infod_init(void) {
    /* bind() in unix_server() needs to be called as root (since in the default
    case we will create the socket in /var/run/), so init_infod() must come
    before setuid(). So it's an extra function. */
    Svfd = unix_server(GOpt.infod_port);

    /* Limit the connections to the user owning the socket (and root). The
    Kernel code for AF_UNIX sockets only requires write permission on the
    socket for that. */
    chown(GOpt.infod_port, GOpt.user_id, 0);
    chmod(GOpt.infod_port, S_IWUSR);
    return;
}

void *infod_main(void *arg) {
    // The INFOD mainloop which waits for incoming connections.
    block_signals();

    while(1) {
        socklen_t sz = sizeof(struct sockaddr_un);
        struct sockaddr_un remote;
        pthread_t id;
        int clfd;

        if((clfd = accept(Svfd, (struct sockaddr *)&remote, &sz)) < 0) {
            continue;
        }
        pthread_create(&id, NULL, client_thread, (void *)clfd);
        pthread_detach(id);
    }

    return NULL;
}

static void *client_thread(void *arg) {
    int fd = (int)arg;

    while(1) {
        static const uint32_t zero = 0;
        unsigned char req;
        uint32_t gint; // general int
        char buf[4096];

        if(!RECEIVE_FULL(fd, &req, sizeof(unsigned char)) || req == IFP_NONE) {
            break;
        } else if(req > IFT_INT_START && req < IFT_INT_STOP) {
            if(!RECEIVE_FULL(fd, &gint, sizeof(uint32_t))) { break; }
            PP_swab(&gint, sizeof(uint32_t));
        } else if(req > IFT_STR_START && req < IFT_STR_STOP) {
            if(!RECEIVE_FULL(fd, &gint, sizeof(uint32_t))) { break; }
            PP_swab(&gint, sizeof(uint32_t));
            if(!RECEIVE_FULL(fd, buf, min_uint(sizeof(buf), gint))) { break; }
        }

        switch(req) {
            // Voids
            case IFP_FLUSH:
                flush_table();
                break;
            case IFP_ZERO:
                zero_counters();
                break;

            // Integers
            case IFP_ACTIVATE ... IFP_DEACTIVSES:
                set_session_status(gint, req);
                break;
            case IFP_REMOVE: {
                struct tty *tty;

                pthread_mutex_lock(&Ttys_lock);
                tty = get_tty(gint);
                pthread_mutex_unlock(&Ttys_lock);

                if(tty != NULL) { log_close(tty); }
                break;
            }
            case IFP_GETINFO:
                getinfo(gint, fd);
                break;
            case IFP_GETINFO_T:
                getinfo_text(gint, fd);
                break;

            // Strings
            // currently none

            // Other
            default:
                notify(LOG_WARNING, NF_NEWLINE, "Unknown request %d, closing"
                 " connection to not run into conversion problems.\n", req);
                goto clthr__exit;
                break;
        }

        send_wait(fd, &zero, sizeof(uint32_t), 0);
    }

clthr__exit:
    close(fd);
    return NULL;
}

//-----------------------------------------------------------------------------
static void flush_table(void) {
    struct HXbtree_node *nd;
    struct HXbtrav *travp;

    pthread_mutex_lock(&Ttys_lock);
    travp = HXbtrav_init(Ttys, NULL);
    while((nd = HXbtraverse(travp)) != NULL) {
        struct tty *tty = nd->data;
        if(tty->status != IFP_DEACTIVATE && tty->fd == -1) {
            HXbtree_del(Ttys, (void *)tty->dev);
            if(tty->file != NULL) { free(tty->file); }
            free(tty);
        }
    }

    pthread_mutex_unlock(&Ttys_lock);
    HXbtrav_free(travp);
    return;
}

static void getinfo_text(uint32_t dev, int fd) {
    if(dev == 0) {
        // No device given, send info about RPLD and all ttys
        struct HXbtree_node *nd;
        struct HXbtrav *travp;

        skprintf(fd,
          "%2s %8s %-7s %9s %9s %4s %s\n"
          "==========================================================================\n"
          "  [---,---] *       %9llu %9llu    -\n"
          "            IOCD: %zu/%zu/%zu/%zu  RW: %zu/%zu  I: %zu  B: %zu\n"
          "--------------------------------------------------------------------------\n",
          "AC", "DEVICE", "TTY", "BYTES IN", "OUT", "FD", "FILENAME",
          Stats.in, Stats.out, Stats.init, Stats.open, Stats.close,
          Stats.deinit, Stats.read, Stats.write, Stats.ioctl, Stats.badpack
        );

        pthread_mutex_lock(&Ttys_lock);
        travp = HXbtrav_init(Ttys, NULL);

        while((nd = HXbtraverse(travp)) != NULL) {
            print_entry_text(fd, nd->data);
        }

        skprintf(fd, "--------------------------------------------------------------------------\n");
        pthread_mutex_unlock(&Ttys_lock);
        HXbtrav_free(travp);
        return;
    } else {
        struct tty *tty;

        pthread_mutex_lock(&Ttys_lock);
        tty = get_tty(dev);
        pthread_mutex_unlock(&Ttys_lock);

        if(tty == NULL) { return; }
        print_entry_text(fd, tty);
    }

    return;
}

static void getinfo(uint32_t dev, int fd) {
    if(dev == 0) {
        // No device given, send info about RPLD and all ttys
        struct HXbtree_node *nd;
        struct HXbtrav *travp;

        skprintf(fd, "%zu %zu %zu %zu %zu %zu %llu %llu %zu %zu\n",
          Stats.init, Stats.open, Stats.close, Stats.deinit, Stats.read,
          Stats.write, Stats.in, Stats.out, Stats.ioctl, Stats.badpack
        );

        pthread_mutex_lock(&Ttys_lock);
        travp = HXbtrav_init(Ttys, NULL);

        while((nd = HXbtraverse(travp)) != NULL) {
            print_entry(fd, nd->data);
        }

        pthread_mutex_unlock(&Ttys_lock);
        HXbtrav_free(travp);
        return;
    } else {
        struct tty *tty;

        pthread_mutex_lock(&Ttys_lock);
        tty = get_tty(dev);
        pthread_mutex_unlock(&Ttys_lock);

        if(tty == NULL) { return; }
        print_entry(fd, tty);
    }

    return;
}

static void set_session_status(uint32_t dev, int req) {
    struct tty *tty;

    pthread_mutex_lock(&Ttys_lock);
    if((tty = get_tty(dev)) == NULL) {
        pthread_mutex_unlock(&Ttys_lock);
        return;
    }

    tty->status = req;
    pthread_mutex_unlock(&Ttys_lock);
    return;
}

static void zero_counters(void) {
    struct HXbtree_node *nd;
    struct HXbtrav *travp;

    pthread_mutex_lock(&Ttys_lock);
    travp = HXbtrav_init(Ttys, NULL);

    while((nd = HXbtraverse(travp)) != NULL) {
        struct tty *tty = nd->data;
        tty->in = tty->out = 0;
    }

    Stats.open = Stats.close = Stats.read = \
    Stats.write = Stats.ioctl = Stats.badpack = 0;
    Stats.in = Stats.out = 0;

    pthread_mutex_unlock(&Ttys_lock);
    HXbtrav_free(travp);
    return;
}

//-----------------------------------------------------------------------------
static void block_signals(void) {
    sigset_t set;
    sigfillset(&set);
    pthread_sigmask(SIG_BLOCK, &set, NULL);
    return;
}

static void print_entry_text(int fd, struct tty *tty) {
    char buf[64], status;

    switch(tty->status) {
        case IFP_ACTIVATE:   status = '*'; break;
        case IFP_DEACTIVATE: status = ' '; break;
        case IFP_DEACTIVSES: status = '.'; break;
        default:             status = '?'; break;
    }

    G_devname_nm(tty->dev, buf, sizeof(buf));
    skprintf(fd, "%c [%3ld,%3ld] %-7s %9zu %9zu %4d %s\n", status,
     KD26_PARTS(tty->dev), buf, tty->in, tty->out, tty->fd, tty->file);
    return;
}

static void print_entry(int fd, struct tty *tty) {
    char buf[64], status = 'x';

    switch(tty->status) {
        case IFP_ACTIVATE:   status = 'A'; break;
        case IFP_DEACTIVATE: status = 'J'; break;
        case IFP_DEACTIVSES: status = 'D'; break;
        default:             status = '?'; break;
    }

    G_devname_nm(tty->dev, buf, sizeof(buf));
    skprintf(fd, "%c %ld %ld %s %zu %zu %d %s\n", status, KD26_PARTS(tty->dev),
     buf, tty->in, tty->out, tty->fd, tty->file);
    return;
}

static int skprintf(int fd, const char *fmt, ...) {
    va_list argp;
    char buf[512];
    uint32_t le, s;

    va_start(argp, fmt);
    le = s = vsnprintf(buf, sizeof(buf), fmt, argp);
    PP_swab(&le, sizeof(size_t));
    send_wait(fd, &le, sizeof(uint32_t), 0);
    send_wait(fd, buf, s, 0);

    va_end(argp);
    return s;
}

static int unix_server(const char *port) {
    struct sockaddr_un sk;
    int fd = -1;

    sk.sun_family = AF_UNIX;
    strncpy(sk.sun_path, port, sizeof(sk.sun_path));
    sk.sun_path[sizeof(sk.sun_path) - 1] = '\0';
    unlink(sk.sun_path);

    if((fd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0 ||
     bind(fd, (struct sockaddr *)&sk, sizeof(struct sockaddr_un)) < 0 ||
     listen(fd, SOMAXCONN) < 0) {
        return -1;
    }

    return fd;
}

//==[ End of file ]============================================================
