/*=============================================================================
ttyrpld - TTY replay daemon
kernel/rpldev.c - Kernel interface for RPLD
  Copyright © Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however only version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <asm/byteorder.h>
#include <asm/semaphore.h>
#include <asm/uaccess.h>
#include <linux/compiler.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/rpl.h>
#include <linux/string.h>
#include <linux/wait.h>
#include <linux/version.h>
#include <linux/vmalloc.h>
#include "../include/rpl_ioctl.h"
#include "../include/rpl_packet.h"

#ifndef __user
#    define __user
#endif
#if __GNUC__ == 3 && __GNUC_MINOR__ >= 4
     /* GCC 3.4 requires functions-to-be-inlined to appear before the call
     to those. I'm not going that way. */
#    define K_INLINE
#else
#    define K_INLINE inline
#endif

#if !(LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0) && \
# LINUX_VERSION_CODE < KERNEL_VERSION(2, 7, 0))
#    error *** Kernel versions other than 2.6.x are not supported
#else
#    define P_MAGIC MAGIC_2_6
#endif

#ifndef KBUILD_BASENAME
#    define KBUILD_BASENAME "rpldev"
#endif
#define PREFIX __stringify(KBUILD_BASENAME) ": "

#ifdef __LITTLE_ENDIAN
#    define PP_swab(ptr)
#else
#    define PP_swab(ptr) my_swab((ptr), sizeof(*(ptr)))
#endif

// Stage 2 functions
static int krn_init(struct tty_struct *, struct tty_struct *);
static int krn_open(struct tty_struct *, struct tty_struct *);
static int krn_read(const char __user *, size_t, struct tty_struct *);
static int krn_write(const char __user *, size_t, struct tty_struct *);
static int krn_close(struct tty_struct *, struct tty_struct *);
static int krn_deinit(struct tty_struct *, struct tty_struct *);
static int krn_ioctl(struct tty_struct *, struct tty_struct *, unsigned int,
  unsigned long);

// Stage 3 functions
static int uif_open(struct inode *, struct file *);
static ssize_t uif_read(struct file *, char __user *, size_t, loff_t *);
static loff_t uif_llseek(struct file *, loff_t, int);
static int uif_ioctl(struct inode *, struct file *, unsigned int,
 unsigned long);
static unsigned int uif_poll(struct file *, poll_table *);
static int uif_close(struct inode *, struct file *);

// Local functions
K_INLINE static ssize_t avail_R(void);
K_INLINE static ssize_t avail_W(void);
K_INLINE static unsigned int imin_uint(unsigned int, unsigned int);
K_INLINE static void get_tty_devnr(uint32_t *, struct tty_struct *);
static int mv_buffer(struct rpld_packet *, const void *, size_t);
K_INLINE static void mv_buffer2(const void *, size_t);
K_INLINE static void mv_to_user(char __user *, size_t);
K_INLINE static void my_swab(void *, size_t);

// Variables
static DECLARE_WAIT_QUEUE_HEAD(Inqueue);
static DECLARE_MUTEX(Open_lock);
static DECLARE_MUTEX(Buffer_lock);
static char *Buffer = NULL, *BufRP = NULL, *BufWP = NULL;
// Each tty has a buffer of 4096, yet the number of possible ttys is endless
static size_t Bufsize = 32768;
static int Minor_nr = MISC_DYNAMIC_MINOR;
static struct file_operations Fops = {
    .open    = uif_open,
    .read    = uif_read,
    .llseek  = uif_llseek,
    .ioctl   = uif_ioctl,
    .poll    = uif_poll,
    .release = uif_close,
    .owner   = THIS_MODULE,
};
static struct miscdevice Misc_info =
 {minor: MISC_DYNAMIC_MINOR, name: "rpl", fops: &Fops};

// Module info
int __init init_module(void);
void __exit cleanup_module(void);
MODULE_DESCRIPTION("ttyrpld Kernel interface");
MODULE_AUTHOR("Jan Engelhardt <jengelh [at] linux01 gwdg de>");
MODULE_LICENSE("GPL v2");
MODULE_PARM(Bufsize, "i");
MODULE_PARM_DESC(Bufsize, "Buffer size (default 32K)");
MODULE_PARM(Minor_nr, "i");
MODULE_PARM_DESC(Minor_nr, "Minor number to use (default: DYNAMIC)");

//-----------------------------------------------------------------------------
int __init init_module(void) {
    int eax;
    Misc_info.minor = Minor_nr;
    if((eax = misc_register(&Misc_info)) != 0) {
        cleanup_module();
        return eax;
    }
    printk(PREFIX "Got registered at minor %d\n", Misc_info.minor);
    return 0;
}

void __exit cleanup_module(void) {
    misc_deregister(&Misc_info);
    return;
}

//-----------------------------------------------------------------------------
static int krn_init(struct tty_struct *this, struct tty_struct *that) {
    struct rpld_packet p;
    p.dev   = p.dev2 = 0;
    if(this != NULL) { get_tty_devnr(&p.dev, this); }
    if(that != NULL) { get_tty_devnr(&p.dev2, that); }
    p.event = EVT_INIT;
    p.magic = P_MAGIC;
    p.size  = 0;
    PP_swab(&p.dev);
    PP_swab(&p.dev2);
    return mv_buffer(&p, NULL, 0);
}

static int krn_open(struct tty_struct *this, struct tty_struct *that) {
    struct rpld_packet p;
    p.dev   = p.dev2 = 0;
    p.event = EVT_OPEN;
    p.magic = P_MAGIC;
    p.size  = 0;

    get_tty_devnr(&p.dev, this);
    if(that != NULL) { get_tty_devnr(&p.dev2, that); }
    PP_swab(&p.dev);
    PP_swab(&p.dev2);
    return mv_buffer(&p, NULL, 0);
}

static int krn_read(const char __user *buf, size_t count,
  struct tty_struct *tty)
{
    /* The data flow is a bit weird at first. krn_read() gets the data on its
    way between ttyDriver(master) -> /dev/stdin(slave), meaning this function
    is called when you hit the keyboard. */
    struct rpld_packet p;
    p.dev2  = 0;
    p.event = EVT_READ;
    p.magic = P_MAGIC;
    p.size  = count;

    get_tty_devnr(&p.dev, tty);
    PP_swab(&p.dev);
    PP_swab(&p.size);
    return mv_buffer(&p, buf, count);
}

static int krn_write(const char __user *buf, size_t count,
  struct tty_struct *tty)
{
    // Data flow: /dev/stdout(slave) -> tty driver(master)
    /* Just as with krn_read() above, .dev2 is not set because it is not used
    at all for EVT_{READ,WRITE,IOCTL}. */
    struct rpld_packet p;
    p.dev2  = 0;
    p.event = EVT_WRITE;
    p.magic = P_MAGIC;
    p.size  = count;

    get_tty_devnr(&p.dev, tty);
    PP_swab(&p.dev);
    PP_swab(&p.size);
    return mv_buffer(&p, buf, count);
}

static int krn_ioctl(struct tty_struct *tty, struct tty_struct *real_tty,
  unsigned int cmd, unsigned long arg)
{
    uint32_t cmd32;
    struct rpld_packet p;

    get_tty_devnr(&p.dev, tty);
    if(real_tty != NULL) {
        get_tty_devnr(&p.dev2, real_tty);
        PP_swab(&p.dev2);
    }
    p.event = EVT_IOCTL;
    p.magic = P_MAGIC;
    p.size  = sizeof(cmd32);
    PP_swab(&p.dev);
    PP_swab(&p.size);
    return mv_buffer(&p, &cmd32, sizeof(cmd32));
}

static int krn_close(struct tty_struct *this, struct tty_struct *that) {
    struct rpld_packet p;
    p.dev   = p.dev2 = 0;
    get_tty_devnr(&p.dev, this);
    PP_swab(&p.dev);
    if(that != NULL) { get_tty_devnr(&p.dev2, that); PP_swab(&p.dev2); }
    p.event = EVT_CLOSE;
    p.magic = P_MAGIC;
    p.size  = 0;
    return mv_buffer(&p, NULL, 0);
}

static int krn_deinit(struct tty_struct *this, struct tty_struct *that) {
    struct rpld_packet p;
    p.dev   = p.dev2 = 0;
    if(this != NULL) { get_tty_devnr(&p.dev, this); }
    if(that != NULL) { get_tty_devnr(&p.dev2, that); }
    p.event = EVT_DEINIT;
    p.magic = P_MAGIC;
    p.size  = 0;
    PP_swab(&p.dev);
    PP_swab(&p.dev2);
    return mv_buffer(&p, NULL, 0);
}

//-----------------------------------------------------------------------------
static int uif_open(struct inode *inode, struct file *filp) {
    // This one is called when the device node has been opened.
    if(inode != NULL) {
        inode->i_mtime = CURRENT_TIME;
        inode->i_mode &= ~(S_IWUGO | S_IXUGO);
    }

    /* The RPL device is read-only. Absolutely. No single O_WRONLY or O_RDWR
    is allowed. */
    if(filp->f_mode & 0x2) { return -EPERM; }

    /* The RPL device can only be opened once, since otherwise, different
    packets could go to different readers. Someone could fork(), though, but
    that's ok. It is just to secure against spurious open()s on the device that
    should not happen, like `find /dev -print0 | xargs -0 grep string`. */
    if(down_trylock(&Open_lock) != 0) { return -EBUSY; }

    Buffer = __vmalloc(Bufsize, GFP_USER | __GFP_HIGHMEM, PAGE_KERNEL);
    if(Buffer == NULL) {
        up(&Open_lock);
        return -ENOMEM;
    }
    BufRP = BufWP = Buffer;

    /* Update links. I do it here 'cause I do not want memory copying (from the
    tty driver to the buffer) when there is no one to read. */
    rpl_qinit   = krn_init;
    rpl_qopen   = krn_open;
    rpl_qread   = krn_read;
    rpl_qwrite  = krn_write;
    rpl_qclose  = krn_close;
    rpl_qioctl  = krn_ioctl;
    rpl_qdeinit = krn_deinit;

    /* The inode's times are changed as follows:
            Access Time: if data is read from the device
      Inode Change Time: when the device is successfully opened
      Modification Time: whenever the device is opened
    */
    if(inode != NULL) { inode->i_ctime = CURRENT_TIME; }
    return 0;
}

static ssize_t uif_read(struct file *filp, char __user *buf,
  size_t count, loff_t *ppos)
{
    // Nothing read, nothing done
    if(count == 0) { return 0; }

    // Must sleep as long as there is no data
    if(down_interruptible(&Buffer_lock)) { return -ERESTARTSYS; }
    while(BufRP == BufWP) {
        up(&Buffer_lock);
        if(filp->f_flags & O_NONBLOCK) { return -EAGAIN; }
        if(wait_event_interruptible(Inqueue, (BufRP != BufWP))) {
            return -ERESTARTSYS;
        }
        if(down_interruptible(&Buffer_lock)) { return -ERESTARTSYS; }
    }

    // Data is available, so give it to the user
    count = imin_uint(count, avail_R());
    mv_to_user(buf, count);

    up(&Buffer_lock);
    filp->f_dentry->d_inode->i_atime = CURRENT_TIME;
    return count;
}

static loff_t uif_llseek(struct file *filp, loff_t offset, int origin) {
    /* Since you can not seek in the circular buffer, the only purpose of
    lseek() is to skip some bytes and "manually" advance the read pointer RP
    (BufRP). Thus, the only accepted origin is SEEK_CUR (1). */

    if(origin != 1) { return -ESPIPE; }
    if(offset == 0) { return 0; }

    if(down_interruptible(&Buffer_lock)) { return -ERESTARTSYS; }
    BufRP += imin_uint(offset, avail_R());
    if(BufRP > Buffer + Bufsize) { BufRP -= Bufsize; }

    up(&Buffer_lock);
    return 0;
}

static int uif_ioctl(struct inode *inode, struct file *filp, unsigned int cmd,
  unsigned long arg)
{
    if(_IOC_TYPE(cmd) != RPL_IOC_MAGIC) { return -ENOTTY; }

    switch(cmd) {
        case RPL_IOC_GETBUFSIZE:
            return Bufsize;
        case RPL_IOC_GETRAVAIL:
            return avail_R();
        case RPL_IOC_GETWAVAIL:
            return avail_W();
        default:
            return -ENOTTY;
    }
    return 0;
}

static unsigned int uif_poll(struct file *filp, poll_table *wait) {
    poll_wait(filp, &Inqueue, wait);
    return POLLIN | POLLRDNORM;
}

static int uif_close(struct inode *inode, struct file *filp) {
    rpl_qinit   = NULL;
    rpl_qopen   = NULL;
    rpl_qread   = NULL;
    rpl_qwrite  = NULL;
    rpl_qioctl  = NULL;
    rpl_qclose  = NULL;
    rpl_qdeinit = NULL;
    vfree(Buffer);
    up(&Open_lock);
    return 0;
}

//-----------------------------------------------------------------------------
K_INLINE static ssize_t avail_R(void) {
    // Return the number of available bytes to read
    if(BufWP >= BufRP) { return BufWP - BufRP; }
    return BufWP + Bufsize - BufRP;
}

K_INLINE static ssize_t avail_W(void) {
    // Return the number of available bytes to write
    if(BufWP >= BufRP) { return BufRP + Bufsize - BufWP - 1; }
    return BufRP - BufWP - 1;
}

K_INLINE static void get_tty_devnr(uint32_t *dev, struct tty_struct *tty) {
    // Get some unique number to identify the thing
    *dev = MKDEV(tty->driver->major, tty->driver->minor_start + tty->index);
    return;
}

K_INLINE static unsigned int imin_uint(unsigned int a, unsigned int b) {
    return a < b ? a : b;
}

static int mv_buffer(struct rpld_packet *p, const void *buf, size_t count) {
    /* Copy the contents the tty driver received to our circulary buffer and
    also add a header (rpld_packet) to it so that the userspace daemon can
    recognize the type. */
    static int Overrun = 0;

    if(down_interruptible(&Buffer_lock)) { return -ERESTARTSYS; }

    if(avail_W() < sizeof(struct rpld_packet) + count) {
        if(!Overrun) {
            printk(KERN_INFO PREFIX "mv_buffer(): Buffer overrun\n");
            ++Overrun;
        }
        up(&Buffer_lock);
        return -ENOSPC;
    } else {
        Overrun = 0;
    }

    mv_buffer2(p, sizeof(struct rpld_packet));
    if(count > 0) { mv_buffer2(buf, count); }
    up(&Buffer_lock);
    wake_up(&Inqueue);
    return count;
}

K_INLINE static void mv_buffer2(const void *src, size_t count) {
    /* This function is responsible for copying (a specific amount of
    arbitrary data) into the circulary buffer. (Taking the wrappage into
    account!) The parent function -- mv_buffer() in this case -- must make sure
    there is enough room. */
    size_t x = Buffer + Bufsize - BufWP;

    if(count <= x) {
        memcpy(BufWP, src, count);
        BufWP += count;
    } else {
        memcpy(BufWP, src, x);
        memcpy(Buffer, src + x, count - x);
        BufWP = Buffer + count - x;
    }

    return;
}

K_INLINE static void mv_to_user(char __user *dest, size_t count) {
    /* mv_to_user() reads COUNT bytes from the circulary buffer and puts it
    into userspace memory. */
    int x = Buffer + Bufsize - BufRP;

    /* Since this operation may block due to userspace page faults, the
    userspace daemon should really have DEST memory-locked. */
    if(count <= x) {
        copy_to_user(dest, BufRP, count);
        BufRP += count;
    } else {
        copy_to_user(dest, BufRP, x);
        copy_to_user(dest + x, Buffer, count - x);
        BufRP = Buffer + count - x;
    }

    return;
}

K_INLINE static void my_swab(void *srcp, size_t count) {
    if(count == 2) {
        uint16_t *ptr = srcp;
        *ptr = cpu_to_le16(*ptr);
    } else if(count == 4) {
        uint32_t *ptr = srcp;
        *ptr = cpu_to_le32(*ptr);
    } else if(count == 8) {
        uint64_t *ptr = srcp;
        *ptr = cpu_to_le64(*ptr);
    }

    return;
}

//==[ End of file ]============================================================
