/*=============================================================================
ttyrpld - TTY replay daemon
user/rdsh.c - Shared functions for RPLD/INFOD/RPLCTL
  Copyright © Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/socket.h>
#include <sys/types.h>
#include <errno.h>
#include <pthread.h>
#include <sched.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>

#include <libHX.h>
#include "dev.h"
#include "global.h"
#include "rdsh.h"

// External Variables
pthread_mutex_t Ttys_lock = PTHREAD_MUTEX_INITIALIZER;
struct HXbtree *Ttys = NULL;
struct Statmap_t Stats;
struct GOptmap_t GOpt = {
    .user_id    = -1,
    .syslog     = 0,
    .verbose    = 0,
    .infod_port = "/var/run/.rplinfod_socket",
    .ofmt       = "%u/%d.%t.%l", // -O
};

//-----------------------------------------------------------------------------
static const char *m_idx = "pqrstuvwxyzabcdef", *s_idx = "0123456789abcdef";

char *G_devname_nm(uint32_t dev, char *buf, size_t count) {
    /* This function returns a "simple name" which can be integrated into
    rpld's log file name scheme. */
    register unsigned long minor = K26_MINOR(dev);

    switch(K26_MAJOR(dev)) {
        case 2: // BSD pty masters
            snprintf(buf, count, "pty%c%c",
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 3: // BSD pty slaves
            snprintf(buf, count, "tty%c%c",
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 4:
            // Virtual console (0-63) and serial console (64-255)
            if(minor < 64) { snprintf(buf, count, "vc-%lu", minor); }
            else { snprintf(buf, count, "ttyS%lu", minor - 64); }
            break;
        case 5:
            if(minor == 1) { snprintf(buf, count, "console"); }
            else { goto unknown; }
        case 128:
            // Unix98 pty masters (got no real /dev entry, though, see below)
            snprintf(buf, count, "ptm-%lu", minor);
            break;
        case 136: // Unix98 pty slaves
            snprintf(buf, count, "pts-%lu", minor);
            break;
        default:
            goto unknown;
    }
    return buf;

unknown:
    /* Not every char device is a tty, and I even do not want to include a
    name database for all possible tty devices. */
    snprintf(buf, count, "%lu-%lu", K26_MAJOR(dev), minor);
    return buf;
}

char *G_devname_fs(uint32_t dev, char *buf, size_t count) {
    /* This function returns a device filename generated from major/minor
    number, which is then used for stat(). */
    register unsigned long minor = K26_MINOR(dev);

    switch(K26_MAJOR(dev)) {
        case 2: // BSD pty masters
            snprintf(buf, count, "/dev/pty%c%c", // devfs: /dev/pty/m%d
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 3: // BSD pty slaves
            snprintf(buf, count, "/dev/tty%c%c", // devfs: /dev/pty/s%d
             m_idx[minor >> 4], s_idx[minor % 16]);
            break;
        case 4:
            // Virtual console (0-63) and serial console (64-255)
            if(minor < 64) { snprintf(buf, count, "/dev/tty%lu", minor); }
            else { snprintf(buf, count, "/dev/ttyS%lu", minor - 64); }
            break;
        case 5:
            if(minor == 1) { snprintf(buf, count, "/dev/console"); }
            else { return NULL; }
        case 128 ... 135:
            /* Unix98 pty masters. They usually have no device nodes in /dev,
            and should instead be accessed over the multiplexer /dev/ptmx.
            However IIRC, devfs popped them up in /dev/ptm. */
            snprintf(buf, count, "/dev/ptm/%lu", minor);
            break;
        case 136 ... 143: // Unix98 pty slaves
            snprintf(buf, count, "/dev/pts/%lu", minor);
            break;
        default:
            return NULL;
    }
    return buf;
}

struct tty *get_tty(uint32_t dev) {
    struct HXbtree_node *ts;
    struct tty *ret = NULL, *tty;

    if((ret = HXbtree_get(Ttys, (void *)dev)) != NULL) { return ret; }
    if((tty = malloc(sizeof(struct tty))) == NULL) { return NULL; }

    tty->dev    = dev;
    tty->uid    = -1;
    tty->fd     = -1;
    tty->status = IFP_ACTIVATE;
    tty->in     = tty->out = 0;
    tty->file   = NULL;

    if((ts = HXbtree_add(Ttys, (void *)dev, tty)) == NULL) {
        free(tty);
        notify(LOG_ERR, "%s: Memory allocation failure\n", __FUNCTION__);
        return NULL;
    }

    return ts->data;
}

void log_close(struct tty *tty) {
    /* Close the logfile and release the tty struct if it does not have special
    options have set. */
    close(tty->fd);
    tty->fd = -1;
    if(tty->file != NULL) {
        free(tty->file);
        tty->file = NULL; // infod
    }

    if(tty->status != IFP_DEACTIVATE) {
        /* If the status is IFP_ACTIVATED, it is reinstantiated upon next
        get_tty(). If it is IFP_DEACTIVSES, it will change to IFP_ACTIVATED,
        as per definition. So we only need the data structure if
        IFP_DEACTIVATED is on. */
        HXbtree_del(Ttys, (void *)tty->dev);
        free(tty);
    }
    return;
}

void notify(int lv, const char *fmt, ...) {
    if(GOpt.verbose) {
        va_list argp;
        va_start(argp, fmt);
        fprintf(stderr, "\n");
        vfprintf(stderr, fmt, argp);
        va_end(argp);
        return; // do not print to syslog if we do to stdout
    }
    if(GOpt.syslog) {
        va_list argp;
        va_start(argp, fmt);
        vsyslog(lv, fmt, argp);
        va_end(argp);
        return;
    }
    return;
}

ssize_t send_wait(int fd, const void *buf, size_t count, int flags) {
    size_t rem = count;
    while(rem > 0) {
        ssize_t eax = send(fd, buf, rem, flags);
        if(eax < 0) { return -errno; }
        if(eax == rem) { break; }
        buf += eax;
        rem -= eax;
        sched_yield();
    }
    return count;
}

//==[ End of file ]============================================================
