/*=============================================================================
ttyrpld - TTY replay daemon
user/rpld.c - User space daemon (filtering, etc.)
  Copyright © Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see doc/GPL2.txt.
=============================================================================*/
#include <sys/mman.h> // mlock()
#include <sys/stat.h>
#include <sys/time.h> // gettimeofday()
#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <popt.h>
#include <pthread.h>
#include <pwd.h>
#include <signal.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
extern int posix_memalign(void **, size_t, size_t);
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include <libHX.h>
#include "dev.h"
#include "rpl_ioctl.h"
#include "rpl_packet.h"
#include "global.h"
#include "rdsh.h"
#define NODEPATH "/var/run/.rpldev"

// The prototype order is the same as the function body order
static void mainloop(int);

static int packet_preprox(struct mem_packet *);
static int packet_process(struct mem_packet *, struct tty *, int);

static void log_cuid(struct tty *);
static int log_open(struct tty *);
static void log_write(struct mem_packet *, struct tty *, int);

static int init_device(const char *);
static int init_sbuffer(int);
static void deinit_sbuffer(void);
static const char *mknod_rpldev(void);
static int init_sighandler(void);
static void sighandler_int(int);
static void sighandler_alrm(int);
static void sighandler_pipe(int);

static int get_options(int *, const char ***);
static int read_config(const char *);
static int read_config_bp(const char *, const char *);
static int pconfig_user(const char *, unsigned char, void *, void *);

inline static const char *basename(const char *);
static char *getnamefromuid(uid_t, char *, size_t);
static uid_t getuidfromname(const char *);
static int value_cmp(void *, void *);

// Global variables (begin with uppercase character)
static void *Buffer = NULL;
static struct {
    int _running, dolog;
    char *ovdev, *fbdev;
    long bsize;

    int infod_start;
} Opt = {
    ._running    = 1,
    .bsize       = 16384,
    .ovdev       = NULL,          // -D
    .fbdev       = "/dev/rpl:/dev/misc/rpl",
    .dolog       = 1,             // !-Q
    .infod_start = 0,             // -I
};

//-----------------------------------------------------------------------------
int main(int argc, const char **argv) {
    pthread_t infod_id;
    int fd;

    umask(umask(0022) | 0477);

    /* Yep, the config file is what is needed by all three
    (rpld, infod, rplctl). */
    if(!read_config("/etc/rpld.conf") && errno != ENOENT) {
        fprintf(stderr, "/etc/rpld.conf exists but could not be read: %s\n",
         strerror(errno));
    }
    if(!read_config_bp(*argv, "rpld.conf") && errno != ENOENT) {
        fprintf(stderr, "$BINPATH/rpld.conf exists but could not be"
         " read: %s\n", strerror(errno));
    }

    if(strcmp(basename(*argv), "rplctl") == 0) {
        return rplctl_main(argc, argv);
    }

    if(!get_options(&argc, &argv)) { return EXIT_FAILURE; }
    memset(&Stats, 0, sizeof(Stats));

    if(GOpt.verbose) {
        printf(
          "# rpld " TTYRPLD_VERSION "\n"
          "This program comes with ABSOLUTELY NO WARRANTY; it is free"
          " software\n"
          "and you are welcome to redistribute it under certain conditions;\n"
          "for details see the doc/GPL2.txt file which should have come with\n"
          "this program.\n\n"
        );
    }

    if((Ttys = HXbtree_init(HXBT_ASSOC | HXBT_LKEY | HXBT_XCPY |
     HXBT_FCMP, value_cmp)) == NULL) {
        perror("Ttys = HXbtree_init()");
        return EXIT_FAILURE;
    }

    if((fd = init_device((Opt.ovdev != NULL) ? Opt.ovdev : Opt.fbdev)) < 0) {
        fprintf(stderr, "No device could be opened, aborting.\n");
        return EXIT_FAILURE;
    }

    if(init_sbuffer(fd) < 0) { return EXIT_FAILURE; }
    init_sighandler();

    // Callback tick for statistics
    if(GOpt.verbose)    { alarm(1); }
    if(Opt.infod_start) { infod_init(); }
    if(GOpt.syslog)     { openlog("rpld", LOG_PID, LOG_DAEMON); }

    // Not seteuid(), because it shall not be posibble to regain privs.
    if(GOpt.user_id >= 0) { setuid(GOpt.user_id); }
    if(Opt.infod_start) { pthread_create(&infod_id, NULL, infod_main, NULL); }
    mainloop(fd);

    if(Opt.infod_start) {
        unlink(GOpt.infod_port);
        pthread_cancel(infod_id);
        pthread_join(infod_id, NULL);
    }

    close(fd);
    deinit_sbuffer();
    return EXIT_SUCCESS;
}

static void mainloop(int fd) {
    while(Opt._running) {
        struct mem_packet packet;
        struct tty *tty;

        if(read(fd, &packet, sizeof(struct rpld_packet)) <
         sizeof(struct rpld_packet)) {
            Opt._running = 0;
            break;
        }

        PP_swab(&packet.dev, sizeof(packet.dev));
        //PP_swab(&packet.dev2, sizeof(packet.dev2)); // .dev2 unused ATM
        PP_swab(&packet.size, sizeof(packet.size));

        if(packet.magic != MAGIC_2_6) {
            ++Stats.badpack;
            notify(LOG_WARNING, "Bogus packet (magic is 0x%02X)!\n",
             packet.magic);
            continue;
        }

        /* Timestamp is taken here instead of in the Kernel module to reduce
        the amount of data which has to be copied over the device. */
        gettimeofday(&packet.tv, NULL);

        if(!packet_preprox(&packet)) {
            G_skip(fd, packet.size, 0);
            continue;
        }

        pthread_mutex_lock(&Ttys_lock);
        if((tty = get_tty(packet.dev)) == NULL) {
            G_skip(fd, packet.size, 0);
            pthread_mutex_unlock(&Ttys_lock);
            continue;
        }

        if(!packet_process(&packet, tty, fd)) {
            /* packet_process is always a success, but it returns 0 to indicate
            if it wnats to skip the payload. */
            G_skip(fd, packet.size, 0);
        }
        pthread_mutex_unlock(&Ttys_lock);
    }

    return;
}

//-----------------------------------------------------------------------------
static int packet_preprox(struct mem_packet *packet) {
    static int *tab[] = {
        [EVT_INIT]   = &Stats.init,
        [EVT_OPEN]   = &Stats.open,
        [EVT_READ]   = &Stats.read,
        [EVT_WRITE]  = &Stats.write,
        [EVT_CLOSE]  = &Stats.close,
        [EVT_DEINIT] = &Stats.deinit,
        [EVT_IOCTL]  = &Stats.ioctl,
        [EVT_max]    = NULL,
    };

    if(packet->event < EVT_max && tab[packet->event] != NULL) {
        ++*tab[packet->event];
    }

    // General packet classification (first stage drop)
    switch(packet->event) {
        // Not used in rpld ATM
        case EVT_INIT:
        case EVT_CLOSE:
        case EVT_IOCTL:
            return 0;

        // These will be processed
        case EVT_OPEN:
        case EVT_DEINIT:
            break;

        // The following roll their own + will be processed...
        case EVT_READ:
            Stats.in += packet->size;
            break;
        case EVT_WRITE:
            Stats.out += packet->size;
            break;
        default:
            notify(LOG_WARNING, "Unknown packet type 0x%02X\n", packet->event);
            return 0;
    }

    {
        register long maj = K26_MAJOR(packet->dev);
        if(maj == 128 /* Unix98 ptm (-) */) {
            if(packet->event != EVT_DEINIT) {
                /* All master side events (except EVT_DEINIT) are dropped,
                since they are pty masters are an exact copy of their slaves,
                so no need to record them too. */
                return 0;
            }

            /* What's up with the ptm/pts stuffage:
            Opening an xterm / screen window (with some debug printf in rpld):
              EVT_INIT   dev=128:5 dev2=136:5 size=0
              EVT_OPEN   dev=136:5 dev2=136:5 size=0

            Closing it:
              EVT_CLOSE  dev=136:5 dev2=128:5 size=0
              EVT_CLOSE  dev=128:5 dev2=136:5 size=0
              EVT_DEINIT dev=128:5 dev2=136:5 size=0

            But killing the screen using ^A^K:
              EVT_CLOSE  dev=128:5 dev2=136:5 size=0
              EVT_CLOSE  dev=136:5 dev2=128:5 size=0
              EVT_DEINIT dev=136:5 dev2=128:5 size=0

            Not all CLOSEs and DEINITs are synchronous. Swapping the devs so
            that dev always is 136: seems to be a good idea. And it works. */

            uint32_t tmp = packet->dev2;
            packet->dev2 = packet->dev;
            packet->dev  = tmp;
        } else if(maj == 2 /* BSD ptm (/dev/ptypXY) */) {
            /* Honestly, I am not sure whether BSD ptms also behave like Unix98
            ptms above, but I assume it. So change major 2 to major 3. */
            if(packet->event != EVT_DEINIT) { return 0; }
            packet->dev = K26_MKDEV(K26_MAJOR(3), K26_MINOR(packet->dev));
        }
    }

    return 1;
}

static int packet_process(struct mem_packet *packet, struct tty *tty, int fd) {
    if(tty->status != IFP_ACTIVATE) { return 0; }

    switch(packet->event) {
        case EVT_OPEN:
            /* EVT_OPEN will not handle logfile opening since log_write() will
            do that when a EVT_{READ,WRITE} pops up. */
            log_cuid(tty);
            break;
        case EVT_READ:
            tty->in += packet->size;
            log_write(packet, tty, fd);
            return 1;
        case EVT_WRITE:
            tty->out += packet->size;
            log_write(packet, tty, fd);
            return 1;
        case EVT_DEINIT:
            log_close(tty);
            break;
        default:
            notify(LOG_ERR, "Should never get here! (%s:%d) Forgot to code"
             " something? (event=%d)\n", __FILE__, __LINE__, packet->event);
            break;
    }

    return 0;
}

//-----------------------------------------------------------------------------
static void log_cuid(struct tty *tty) {
    struct stat sb;
    char buf[32];

    if(tty->uid == -1) { return; }
    G_devname_fs(tty->dev, buf, sizeof(buf));
    if(stat(buf, &sb) == 0 && sb.st_uid != tty->uid) {
        tty->in = tty->out = 0;
        log_open(tty);
    }
    return;
}

static int check_parent_directory(const char *s) {
    char *path = alloca(strlen(s) + 1), *p;
    strcpy(path, s);
    if((p = strrchr(path, '/')) == NULL) {
        // Current dirctory, no more dir checks needed
        return 1;
    }
    *p = '\0';
    return HX_mkdir(path); // like `mkdir -p`
}

static int log_open(struct tty *tty) {
    char buf[MAXFNLEN], devnode[32], fmday[16], fmtime[16], user[32];
    struct HX_repmap catalog[] = {{'d', NULL, fmday}, {'l', NULL, devnode},
     {'t', NULL, fmtime}, {'u', NULL, user}, {0}};

    { // User
      struct stat sb;
      G_devname_fs(tty->dev, devnode, sizeof(devnode));
      if(stat(devnode, &sb) != 0) {
          strcpy(user, "NONE");
      } else {
          tty->uid = sb.st_uid;
          if(getnamefromuid(sb.st_uid, user, sizeof(user)) == NULL) {
              snprintf(user, sizeof(user), "%d", sb.st_uid);
          }
      }
    }

    { // Time
      time_t now = time(NULL);
      struct tm now_tm;
      localtime_r(&now, &now_tm);
      strftime(fmday,  sizeof(fmday),  "%Y%m%d", &now_tm);
      strftime(fmtime, sizeof(fmtime), "%H%M%S", &now_tm);
    }

    // Simple tty name
    G_devname_nm(tty->dev, devnode, sizeof(devnode));
    HX_strrep(buf, MAXFNLEN, GOpt.ofmt, catalog);

    // Open log
    HX_strclone(&tty->file, buf);
    if(!Opt.dolog) {
        snprintf(buf, MAXFNLEN, "/dev/null");
    } else if(check_parent_directory(buf) <= 0) {
        notify(LOG_ERR, "Directory permission denied: It won't be possible to"
         " write to the file %s, expect warnings.\n", buf);
    }
    if((tty->fd = open(buf, O_WRONLY | O_CREAT | O_APPEND, 0200)) < 0) {
        notify(LOG_ERR, "Could not open %s: %s\n", buf, strerror(errno));
    }

    // Ident header (optional)
    if(tty->fd >= 0) {
        struct disk_packet p = {
            .event = EVT_IDENT,
            .magic = MAGIC_SIG,
            .tv    = {-1, -1},
        };
        size_t s;

        buf[MAXFNLEN - 1] = '\0';
        strncpy(buf, "ttyrpld " TTYRPLD_VERSION, MAXFNLEN - 1);
        s = p.size = strlen(buf);
        PP_swab(&p.size, sizeof(p.size));
        write(tty->fd, &p, sizeof(struct disk_packet));
        write(tty->fd, buf, s);
    }

    return tty->fd;
}

static void log_write(struct mem_packet *packet, struct tty *tty, int fd) {
    /* Slap a timestamp around the packet and write it out to disk. Open the
    log if necessary. */
    char *buffer = alloca(packet->size);
    int have;

    if(tty->fd == -1) { log_open(tty); }
    if((have = read(fd, buffer, packet->size)) <= 0) { return; }
    if(have != packet->size) { packet->size = have; }

    PP_swab(&packet->size, sizeof(packet->size));
    PP_swab(&packet->tv.tv_sec, sizeof(packet->tv.tv_sec));
    PP_swab(&packet->tv.tv_usec, sizeof(packet->tv.tv_usec));
    write(tty->fd, &packet->size, sizeof(struct disk_packet));
    write(tty->fd, buffer, have);
    return;
}

//-----------------------------------------------------------------------------
static int init_device(const char *in_devs) {
    const char *new_dev = NULL;
    int fd = -1;

    if(Opt.ovdev == NULL && (new_dev = mknod_rpldev()) != NULL) {
        if((fd = open(new_dev, O_RDONLY)) == -1) {
            int se = errno;
            new_dev = NULL;
            perror("dynamic_find: Could not connect to RPL device");
            if(se == EBUSY) {
                fprintf(stderr, "\t" "The RPL device can only be opened once,"
                 "\n\t" "there is probably an instance of rpld running!\n");
            }
        } else if(GOpt.verbose) {
            printf("Connected to RPL device\n");
        }
        unlink(NODEPATH);
        unlink(new_dev);
    }

    if(new_dev == NULL) {
        /* No, this will not change into an else if, because new_dev might be
        set to something else above! */
        char *copy = HX_strdup(in_devs), *devs = copy, *devp;

        while((devp = HX_strsep(&devs, ":")) != NULL) {
            if((fd = open(devp, O_RDONLY)) >= 0) {
                if(GOpt.verbose) { printf("Connected to %s\n", devp); }
                break;
            }
            if(errno != ENOENT) {
                fprintf(stderr, "static_find: Could not open %s even though "
                 "it exists: %s (trying next device)\n",
                 devp, strerror(errno));
            }
            if(errno == EBUSY) {
                fprintf(stderr, "\t" "The RPL device can only be opened once,"
                 "\n\t" "there is probably an instance of rpld running!\n");
            }
            ++devp;
        }
        free(copy);
    }

    return fd;
}

static int init_sbuffer(int fd) {
    int eax;

    if((eax = ioctl(fd, RPL_IOC_GETBUFSIZE, 0)) > 0) {
        Opt.bsize = eax;
    }

    if(posix_memalign(&Buffer, sysconf(_SC_PAGESIZE), Opt.bsize) != 0) {
        eax = errno;
        perror("malloc()/posix_memalign()");
        return -(errno = eax);
    }

    /* Done so that this memory area can not be swapped out, which is essential
    against buffer overruns. */
    mlock(Buffer, Opt.bsize);
    return 1;
}

static void deinit_sbuffer(void) {
    munlock(Buffer, Opt.bsize);
    free(Buffer);
    return;
}

static const char *mknod_rpldev(void) {
    int32_t minor = -1;
    char buf[128];
    FILE *fp;

    if((fp = fopen("/proc/misc", "r")) == NULL) {
        perror("Could not open /proc/misc");
        return NULL;
    }

    while(fgets(buf, sizeof(buf), fp) != NULL) {
        char *ptr = buf, *name;
        while(!isdigit(*ptr)) { ++ptr; }
        minor = strtol(ptr, NULL, 0);
        while(isdigit(*ptr)) { ++ptr; }
        while(isspace(*ptr)) { ++ptr; }

        name = ptr;
        while(isalnum(*ptr)) { ++ptr; }
        *ptr = '\0';
        if(strcmp(name, "rpl") == 0) { break; }
        minor = -1;
    }

    fclose(fp);

    if(minor == -1) {
        printf("RPL module does not seem to be loaded\n");
        return NULL;
    }

    if(mknod(NODEPATH, S_IFCHR | 0400, GLIBC_MKDEV(10, minor)) != 0) {
        return NULL;
    }

    chown(NODEPATH, GOpt.user_id, 0);
    return NODEPATH;
}

static int init_sighandler(void) {
    struct sigaction s_int, s_alrm, s_pipe;

    s_int.sa_handler = sighandler_int;
    s_int.sa_flags   = SA_RESTART;
    sigemptyset(&s_int.sa_mask);

    s_alrm.sa_handler = sighandler_alrm;
    s_alrm.sa_flags   = SA_RESTART;
    sigemptyset(&s_alrm.sa_mask);

    s_pipe.sa_handler = sighandler_pipe;
    s_pipe.sa_flags   = SA_RESTART;
    sigemptyset(&s_pipe.sa_mask);

    /* All sigactions() shall be executed, however, if one fails, this function
    shall return <= 0, otherwise >0 upon success.
    Geesh, I love these constructs. */
    return !(!!sigaction(SIGINT, &s_int, NULL) + 
     !!sigaction(SIGTERM, &s_int, NULL) +
     !!sigaction(SIGALRM, &s_alrm, NULL) +
     !!sigaction(SIGPIPE, &s_pipe, NULL));
}

static void sighandler_int(int s) {
    if(Opt._running-- == 0) {
        if(GOpt.verbose) {
            printf("Second time we received SIGINT/SIGTERM,"
             " exiting immediately.\n");
        }
        exit(EXIT_FAILURE);
    }
    if(GOpt.verbose) {
        printf("\n" "Received SIGINT/SIGTERM, shutting down.\n");
    }
    Opt._running = 0;
    return;
}

static void sighandler_alrm(int s) {
    fprintf(stderr, "\r\e[2K" "IOCD: %zu/%zu/%zu/%zu  RW: %zu/%zu (%llu/%llu)"
      " I: %zu  B: %zu",
      Stats.init, Stats.open, Stats.close, Stats.deinit, Stats.read, Stats.write, Stats.in,
      Stats.out, Stats.ioctl, Stats.badpack
    );
    if(GOpt.verbose) { alarm(1); }
    return;
}

static void sighandler_pipe(int s) {
    fprintf(stderr, "\n" "[%d] Received SIGPIPE\n", getpid());
    return;
}

//-----------------------------------------------------------------------------
static int get_options(int *argc, const char ***argv) {
    static const char *_empty_argv[] = {NULL};
    char *tmp;
    struct poptOption options_table[] = {
        {NULL, 'D', POPT_ARG_STRING, &Opt.ovdev, 0,
         "Path to the RPL device", "file"},
        {NULL, 'I', POPT_ARG_VAL, &Opt.infod_start, 1,
         "Start INFOD subcomponent", NULL},
        {NULL, 'O', POPT_ARG_STRING, &GOpt.ofmt, 0,
         "Override OFMT variable", "string"},
        {NULL, 'Q', POPT_ARG_NONE, NULL, 'Q',
         "Deactivate logging, only do bytecounts", NULL},
        {NULL, 'U', POPT_ARG_STRING, &tmp, 'U',
         "User to change to", "user"},
        {NULL, 'c', POPT_ARG_STRING, &tmp, 'c',
         "Read configuration from file (on top of hardcoded conf)", "file"},
        {NULL, 'i', POPT_ARG_VAL, &Opt.infod_start, 0,
         "Do not start INFOD subcomponent", NULL},
        {NULL, 's', POPT_ARG_NONE, &GOpt.syslog, 0,
         "Print warnings/errors to syslog", NULL},
        {NULL, 'v', POPT_ARG_NONE, &GOpt.verbose, 0,
         "Print statistics while rpld is running (overrides -s)", NULL},
        POPT_AUTOHELP
        POPT_TABLEEND
    };

    poptContext ctx;
    const char **args;
    int c, argk;

    ctx = poptGetContext(**argv, *argc, *argv, options_table, 0);
    while((c = poptGetNextOpt(ctx)) >= 0) {
        switch(c) {
            case 'Q':
                Opt.dolog = 0;
                break;
            case 'U':
                if(!pconfig_user(NULL, 0, tmp, NULL)) { return 0; }
                free(tmp);
                break;
            case 'c':
                read_config(tmp);
                free(tmp);
                break;
        }
    }

    if(c < -1) {
        fprintf(stderr, "%s: %s\n", poptBadOption(ctx, 0), poptStrerror(c));
        poptFreeContext(ctx);
        return 0;
    }

    if((args = poptGetArgs(ctx)) != NULL) {
        argk = count_args(args);
        poptDupArgv(argk, args, argc, argv);
    } else {
        *argc = 0;
        *argv = _empty_argv;
    }

    poptFreeContext(ctx);
    return 1;
}

static int read_config(const char *file) {
    static struct rconfig_opt options_table[] = {
        {"OVDEVICE", RCONF_STRING, &Opt.ovdev, NULL},
        {"FBDEVICE", RCONF_STRING, &Opt.fbdev, NULL},
        {"OFMT",     RCONF_STRING, &GOpt.ofmt, NULL},
        {"DO_LOG",   RCONF_IBOOL,  &Opt.dolog, NULL},
        {"USER",     RCONF_CB,     NULL,       pconfig_user},

        {"INFOD_PORT", RCONF_STRING, &GOpt.infod_port, NULL},
        {"START_INFOD", RCONF_IBOOL, &Opt.infod_start, NULL},
        {NULL},
    };
    return HX_rconfig(file, options_table);
}

static int read_config_bp(const char *app_path, const char *file) {
    char *fpath = HX_strdup(app_path), *ptr, construct[MAXFNLEN];
    if((ptr = strrchr(fpath, '/')) == NULL) {
        construct[MAXFNLEN - 1] = '\0';
        strncpy(construct, file, MAXFNLEN - 1);
    } else {
        *ptr++ = '\0';
        snprintf(construct, MAXFNLEN, "%s/%s", fpath, file);
    }
    free(fpath);
    return read_config(construct);
}

static int pconfig_user(const char *key, unsigned char type, void *ptr,
 void *uptr) {
    if((GOpt.user_id = getuidfromname(ptr)) < 0) {
        fprintf(stderr, "No such user: %s\n", (const char *)ptr);
        exit(EXIT_FAILURE);
    }
    return 1;
}

//-----------------------------------------------------------------------------
inline static const char *basename(const char *s) {
    // Return file component of a full-path filename
    const char *p;
    if((p = strrchr(s, '/')) != NULL) { return p + 1; }
    return s;
}

static char *getnamefromuid(uid_t uid, char *result, size_t len) {
    struct passwd ent, *ep;
    char additional[1024];
    getpwuid_r(uid, &ent, additional, sizeof(additional), &ep);
    if(ep == NULL) { return NULL; }
    strncpy(result, ep->pw_name, len - 1);
    result[len - 1] = '\0';
    return result;
}

static uid_t getuidfromname(const char *name) {
    struct passwd ent, *ep;
    char additional[1024];
    getpwnam_r(name, &ent, additional, sizeof(additional), &ep);
    if(ep == NULL) { return -1; }
    return ep->pw_uid;
}

static int value_cmp(void *pa, void *pb) {
    return (pa > pb) - (pa < pb);
}

//==[ End of file ]============================================================
