/*=============================================================================
ttyrpld - TTY replay daemon
user/replay.c - Realtime TTY log analyzer
  Copyright © Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL v2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see the COPYING file
=============================================================================*/
#include <sys/stat.h>
#include <sys/time.h> // usec def, gettimeofday()
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <popt.h>
#include <sched.h> // sched_yield()
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h> // nsec def, nanosleep()
#include <unistd.h>

#include "dev.h"
#include "rpl_packet.h"
#include "global.h"
#include "pctrl.h"

#define offsetof(type, member) ((size_t)&((type *)NULL)->member)
#define containerof(var, type, member) ((void *)(var) - offsetof(type, member))
#define MICROSECOND 1000000
#define NANOSECOND  1000000000
#define BUFSIZE     4096

// Structure definitions
enum { // current follow mode state:
    FM_NONE = 0,
    FM_CATCHUP, // playing old content
    FM_LIVE,    // already hit EOF once and waiting for new data
    FM_NODELAY, // same as FM_LIVE, but different code parts are executed
};

struct {
    double factor;
    int follow, nodelay, sktype, no_pctrl, showtime;
    long ovcorr, maxdelay, skval;
} Opt = {
    .factor   = 1.0,        // -S
    .follow   = 0,          // -F/-f
    .maxdelay = -1,         // -m
    .nodelay  = 0,          // -T
    .sktype   = PCTRL_NONE, // -J/-j
    .skval    = 0,
    .showtime = 0,          // -t
    .no_pctrl = 0,          // --no-pctrl
};

// Functions
static int replay_file(int, const char *);
static void e_proc(int, struct disk_packet *, struct pctrl_info *, char *,
  struct timeval *, unsigned long *);

static unsigned long calc_ovcorr(unsigned long, int);
static int find_next_packet(int);
static int get_options(int *, const char ***);
inline static ssize_t read_nullfm(int, size_t);
static ssize_t read_through(int, int, size_t);
static ssize_t read_waitfm(int, void *, size_t);
static int seek_to_end(int);
static int usleep_ovcorr(struct timeval *, long *);
static void tv_delta(const struct timeval *, const struct timeval *,
 struct timeval *);
inline static unsigned long long tv2usec(const struct timeval *);
inline static void usec2tv(unsigned long long, struct timeval *);

//-----------------------------------------------------------------------------
int main(int argc, const char **argv) {
    const char *auto_stdin[] = {"/dev/stdin", NULL};
    int i;

    if(!get_options(&argc, &argv)) { return EXIT_FAILURE; }

    fprintf(stderr, "> ttyreplay " TTYRPLD_VERSION "\n");
    fprintf(stderr,
      "This program comes with ABSOLUTELY NO WARRANTY; it is free "
      "software\n" "and you are welcome to redistribute it under certain "
      "conditions; for\n" "details see the COPYING file which should have "
      "come with this program.\n"
    );

    if(argc == 0) {
        if(!isatty(STDIN_FILENO)) {
            fprintf(stderr, "No arguments given but STDIN seems to be a "
              "pipe/file -- replaying from STDIN\n");
            argv = auto_stdin;
        } else {
            fprintf(stderr, "No filename(s) given.\n");
            exit(EXIT_FAILURE);
        }
    }

    setvbuf(stdout, NULL, _IONBF, 0);
    Opt.ovcorr = calc_ovcorr(0, 100);
    if(!Opt.no_pctrl) { pctrl_init(); }

    for(i = 0; i < argc; /* see switch() */) {
        int fd, rv;
        if(strcmp(argv[i], "-") == 0) { argv[i] = "/dev/stdin"; }
        if((fd = open(argv[i], O_RDONLY)) < 0) {
            fprintf(stderr, "Could not open %s: %s\n",
              argv[i], strerror(errno));
            ++i;
            continue;
        }

        rv = replay_file(fd, argv[i]);
        close(fd);

        switch(rv) {
            case PCTRL_PREV:
                if(i > 0) { --i; }
                break;
            case PCTRL_NEXT:
                if(i < argc - 1) { ++i; }
                break;
            case PCTRL_EXIT:
                i = argc;
                break;
            default:
                ++i;
                break;
        }
    }

    pctrl_exit();
    return EXIT_SUCCESS;
}

static int replay_file(int fd, const char *name) {
    struct disk_packet packet;
    struct pctrl_info ps;
    struct timeval stamp;
    ssize_t eax;
    long skew = 0;
    char tick = 0;

    if(Opt.follow == FM_LIVE && !seek_to_end(fd)) { return 0; }

    memset(&ps, 0, sizeof(struct pctrl_info));
    ps.echo   = 0;
    ps.factor = Opt.factor;
    ps.sktype = Opt.sktype;
    ps.skval  = Opt.skval;
    pctrl_activate(&ps);

    while((eax = read_waitfm(fd, &packet, sizeof(struct disk_packet))) ==
     sizeof(struct disk_packet)) {
        if(packet.magic != MAGIC_SIG) {
            fprintf(stderr, "\n" "<Packet inconsistency! "
             "Trying to find next valid packet.>\n");
            tick = 0;
            /* If read() in find_next_packet() generates an error, it will be
            catched when the condition is re-evaulated upon continue. */
            find_next_packet(fd);
            continue;
        }

        SWAB(&packet.size, sizeof(packet.size));
        SWAB(&packet.tv.tv_sec, sizeof(packet.tv.tv_sec));
        SWAB(&packet.tv.tv_usec, sizeof(packet.tv.tv_usec));

        switch(packet.event) {
            case EVT_WRITE:
                e_proc(fd, &packet, &ps, &tick, &stamp, &skew);
                break;
            case EVT_IDENT:
                fprintf(stderr, "\n\e[1;37m%s ", "Created using"); // ]
                read_through(fd, STDERR_FILENO, packet.size);
                fprintf(stderr, "\e[0m\n"); // ]
                break;
            case EVT_DEINIT:
                fprintf(stderr, "\n\e[1;37;41m%s\e[0m\n", // ]]
                 "<tty device has been closed>");
                // fallthrough kinda
                read_nullfm(fd, packet.size);
                break;
            case EVT_READ:
                if(ps.echo) {
                    printf("\e[1;37;41m"); // ]
                    e_proc(fd, &packet, &ps, &tick, &stamp, &skew);
                    printf("\e[0m"); // ]
                    break;
                }
                // true fallthrough
            default:
                read_nullfm(fd, packet.size);
                break;
        }

        if(ps.brk) { break; }
    }

    pctrl_deactivate(0);

    if(eax < 0) {
        fprintf(stderr, "\n\e[1;37;41m%s\e[0m\nread(): %s\n", // ]]
         "<Error while reading from stream>", strerror(errno));
    }

    printf("\n\e[1;37;41m"); // ]
    printf("<Log replaying of %s finished>", name);
    printf("\e[0m\n"); // ]
    return ps.brk;
}

static void e_proc(int fd, struct disk_packet *p, struct pctrl_info *i,
  char *tick, struct timeval *stamp, unsigned long *skew)
{
    if(i->sktype == PCTRL_SKPACK && i->skval-- > 0) {
        // Just skip, do not update tick/stamp/skew/delta
        read_through(fd, STDOUT_FILENO, p->size);
        return;
    }

    if(!*tick) {
        // No delay after the first packet has been read...
        ++*tick;
        if(i->skval > 0) {
            memcpy(stamp, &p->tv, sizeof(struct timeval));
            read_through(fd, STDOUT_FILENO, p->size);
            return;
        }
    } else if(Opt.follow != FM_LIVE && Opt.follow != FM_NODELAY) {
        /* ... only when we know the next packet (which is already read by now)
        the tiem difference can be calculated. This code is not executed in
        FM_LIVE mode, because we have been already waiting enough with
        read_wait(). */
        struct timeval delta;
        tv_delta(stamp, &p->tv, &delta);

        if(i->sktype == PCTRL_SKTIME &&
         (i->skval -= tv2usec(&delta) / 1000) > 0) {
            memcpy(stamp, &p->tv, sizeof(struct timeval));
            read_through(fd, STDOUT_FILENO, p->size);
            return;
        }
        if(i->factor != 1.0) {
            // -S switch for the fast
            usec2tv((double)tv2usec(&delta) / i->factor, &delta);
        }
        if(Opt.maxdelay > 0 && tv2usec(&delta) > Opt.maxdelay) {
            // -m switch for those who do not want to wait their lifetime.
            usec2tv(Opt.maxdelay, &delta);
        }
        usleep_ovcorr(&delta, skew);
    }

    while(i->paused) { sched_yield(); }

    memcpy(stamp, &p->tv, sizeof(struct timeval));
    read_through(fd, STDOUT_FILENO, p->size);

    if(Opt.showtime) {
        struct tm tm;
        char buf[80];
        time_t now = p->tv.tv_sec;
        localtime_r(&now, &tm);
        if(Opt.showtime == 1) {
            strftime(buf, sizeof(buf), "\e7\e[1;300H\e[9D" // ]]
             "\e[0;7m[\e[0;1;37;41m" "%H:%M:%S" "\e[0;7m]\e8", &tm); // ]]]
        } else {
            strftime(buf, sizeof(buf), "\e7\e[1;300H\e[12D" // ]]
             "\e[0;7m[\e[0;1;37;41m" "%d.%m %H:%M" "\e[0;7m]\e8", &tm); // ]]]
        }
        write(STDOUT_FILENO, buf, strlen(buf));
    }
    return;
}

//-----------------------------------------------------------------------------
static unsigned long calc_ovcorr(unsigned long ad, int rd) {
    struct timespec s = {.tv_sec = 0, .tv_nsec = ad};
    struct timeval start, stop;
    unsigned long av = 0;
    int count = rd;

    fprintf(stderr, "Calculating average overhead...");

    while(count--) {
        gettimeofday(&start, NULL);
        nanosleep(&s, NULL);
        gettimeofday(&stop, NULL);
        av += MICROSECOND * (stop.tv_sec - start.tv_sec) +
         stop.tv_usec - start.tv_usec;
    }

    av /= rd;
    fprintf(stderr, " %lu µs\n", av);
    return av;
}

static int find_next_packet(int fd) {
    /* There is more algorithmic code than simple code in ttyreplay.
    Outstanding. */
#define LZ           sizeof(struct disk_packet)
#define BZ           (2 * LZ)
#define MAGIC_OFFSET offsetof(struct disk_packet, magic)
    char buf[BZ];
    struct disk_packet *packet = (void *)buf;
    size_t s;
    int ok = 0;

    if(read_waitfm(fd, buf, BZ) < BZ) { return 0; }

    while(1) {
        char *ptr;

        /* Indeed, the many read() calls get more and more data from the
        stream without displaying it. Anyway, if we found a way into this
        function, there is a reason to it. */

        if((ptr = memchr(buf, MAGIC_SIG, BZ)) != NULL &&
         ptr - buf >= MAGIC_OFFSET) {
            // A magic byte has been found and the packet start is complete
            char *ctx = containerof(ptr, struct disk_packet, magic);
            if(ctx != buf) {
                size_t cnt = buf + BZ - ctx;
                ctx = memmove(buf, ctx, cnt);
                if(read_waitfm(fd, buf + cnt, BZ - cnt) < BZ - cnt) {
                    return 0;
                }
            }
        } else if(ptr != NULL) {
            /* Magic byte, but of no use. Discard it, and fill up with new
            data from the descriptor. */
            size_t cnt = buf + BZ - ptr - 1;
            memmove(buf, ptr + 1, cnt);
            if(read_waitfm(fd, buf + cnt, BZ - cnt) < BZ - cnt) { return 0; }
            continue;
        } else {
            /* No magic byte, but since it might be just the next byte in
            the stream, only read LZ bytes. */
            memmove(buf, buf + BZ - LZ, BZ - LZ);
            if(read_waitfm(fd, buf + LZ, BZ - LZ) < BZ - LZ) { return 0; }
            continue;
        }

        s = packet->size;
        if(s > 4096) {
            /* The default tty buffer size is 4096, and any size above this is
            questionable at all. Start with a new slate. */
            ok = 0;
            memmove(buf, buf + LZ, BZ - LZ);
            if(read_waitfm(fd, buf, BZ) < BZ) { return 0; }
            continue;
        }

        if(s < LZ) {
            memmove(buf, buf + LZ + s, BZ - LZ - s);
            if(read_waitfm(fd, buf + BZ - LZ - s, LZ + s) < LZ + s) {
                return 0;
            }
        } else {
            /* There is no header (according to .size) in our buffer, so we
            can blindly munge lots of data. */
            if(!read_nullfm(fd, s - LZ) || read_waitfm(fd, buf, BZ) < BZ) {
                return 0;
            }
        }

        if((ptr = memchr(buf, MAGIC_SIG, MAGIC_OFFSET + 1)) == NULL ||
         ptr - buf != MAGIC_OFFSET) {
            /* If the size field does not match up with the next magic byte,
            drop it all. */
            ok = 0;
            continue;
        }

        if(++ok >= 2) { break; }
    }

    fprintf(stderr, "\n" "<Found packet boundary>\n");

    // Finally adjust the read pointer to a packet boundary
    if((s = packet->size) < 16) {
        /* Mmhkay, there is another header other than the current (packet)
        in the buffer. Crap, another one gone. */
        memmove(buf, buf + LZ + s, BZ - LZ - s);
        if(read_waitfm(fd, buf + BZ - LZ - s, s) < s ||
         !read_nullfm(fd, packet->size)) {
            return 0;
        }
    } else {
        // Just subtract what we have already read into the buffer
        if(!read_nullfm(fd, s - LZ)) { return 0; }
    }

    return 1;
#undef BZ
#undef LZ
#undef MAGIC_OFFSET
}

static int get_options(int *argc, const char ***argv) {
    static const char *_empty_argv[] = {NULL, NULL};
    char *tmp = NULL;
    struct poptOption options_table[] = {
        {"no-pctrl", 0, POPT_ARG_NONE, &Opt.no_pctrl, 0,
         "Do not enable playing controls", NULL},
        {NULL, 'F', POPT_ARG_NONE, NULL, 'F',
         "Live feed follow mode (like `tail -f`)", NULL},
        {NULL, 'J', POPT_ARG_STRING, &tmp, 'J',
         "Seek to time position (HH:MM:SS or total SEC, "
         "see docs)", "tspec"},
        {NULL, 'S', POPT_ARG_DOUBLE, &Opt.factor, '\0',
         "Speed factor (default: 1.0)", NULL},
        {NULL, 'T', POPT_ARG_NONE, &Opt.nodelay, 'T',
         "No delays (useful for text file grepping)", NULL},
        {NULL, 'f', POPT_ARG_NONE, NULL, 'f',
         "Catch-up follow mode (play file, switch to live feed on EOF)", NULL},
        {NULL, 'j', POPT_ARG_LONG, &Opt.skval, 'j',
         "Skip the given number of EVT_WRITE packets", NULL},
        {NULL, 'm', POPT_ARG_LONG, &Opt.maxdelay, 'm',
         "Maximum delay to wait between packets", "msec"},
        {NULL, 't', POPT_ARG_NONE, NULL, 't',
         "Show the timestamp at the top of the screen", NULL},
        POPT_AUTOHELP
        POPT_TABLEEND
    };

    poptContext ctx;
    const char **args;
    int c, argk;

    ctx = poptGetContext(**argv, *argc, *argv, options_table, 0);
    while((c = poptGetNextOpt(ctx)) >= 0) {
        switch(c) {
            case 'F':
                Opt.follow = FM_LIVE;
                break;
            case 'J': { // jump to time
                char *sec = "", *min = "", *hr = "";

                Opt.sktype = PCTRL_SKTIME;
                Opt.skval  = 0;

                if((sec = strrchr(tmp, ':')) == NULL) {
                    sec = tmp;
                } else {
                    *sec++ = '\0';
                    if((min = strrchr(tmp, ':')) == NULL) {
                        min = tmp;
                    } else {
                        *min++ = '\0';
                        hr = tmp;
                    }
                }
                Opt.skval = (strtol(sec, NULL, 0) + 60 * strtol(min, NULL, 0) +
                 3600 * strtol(hr, NULL, 0)) * 1000;
                free(tmp);
                break;
            }
            case 'T':
                Opt.follow = FM_NODELAY;
                break;
            case 'f':
                Opt.follow = FM_CATCHUP;
                break;
            case 'j': // jump to packet
                Opt.sktype = PCTRL_SKPACK;
                break;
            case 'm': // maximum delay between two packets
                Opt.maxdelay *= 1000;
                break;
            case 't': // Show timestamp
                ++Opt.showtime;
                break;
        }
    }

    if(c < -1) {
        fprintf(stderr, "%s: %s\n", poptBadOption(ctx, 0), poptStrerror(c));
        poptFreeContext(ctx);
        return 0;
    }

    if((args = poptGetArgs(ctx)) != NULL) {
        argk = count_args(args);
        poptDupArgv(argk, args, argc, argv);
    } else {
        *argc = 0;
        *argv = _empty_argv;
    }

    poptFreeContext(ctx);
    return 1;
}

static ssize_t read_through(int in, int out, size_t count) {
    // Read from IN and directly give it to OUT. This is like sendfile().
    char buf[BUFSIZE];
    size_t rem = count;

    while(rem > 0) {
        ssize_t eax = read(in, buf, min_uint(BUFSIZE, rem));
        if(eax < 0) { return 0; }
        write(out, buf, eax);
        if(eax == rem) { break; }
        rem -= eax;
        sched_yield();
    }

    return count;
}

inline static ssize_t read_nullfm(int fd, size_t count) {
    return G_skip(fd, count, Opt.follow != FM_NONE &&
     Opt.follow != FM_NODELAY);
}

static ssize_t read_waitfm(int fd, void *buf, size_t count) {
    /* Wrapper function for either read() or read_wait(). If EOF has been
    detected (ATM), we switch into FM_LIVE mode (if currently in FM_CATCHUP
    mode). */
    struct stat sb;
    off_t pos;

    if(Opt.follow == FM_NONE || Opt.follow == FM_NODELAY) {
        /* If no follow mode is selected, the while() loop in replay_file()
        shall terminate as soon as we encounter EOF or have a short read (i.e.
        read less then count). */
        return read(fd, buf, count);
    }

    /* If it is a live feed (follow mode), complete the read (i.e. read all
    requested bytes. fstat() must come after lseek() -- in the very case that
    the file size increases inbetween. */
    if(Opt.follow == FM_CATCHUP && (pos = lseek(fd, 0, SEEK_CUR)) != 0 &&
     fstat(fd, &sb) == 0 && pos == sb.st_size) {
        fprintf(stderr, "\n" "<Switching to live feed follow mode>\n");
        Opt.follow = FM_LIVE;
    }

    return read_wait(fd, buf, count);
}

static int seek_to_end(int fd) {
    if(lseek(fd, 0, SEEK_END) == -1 && errno == ESPIPE) {
        /* Some workaround for non-seekable descriptors. G_skip() is not
        suitable, since ... it's just not designed to do what is below, i.e.
        non-blocking-until-EOF-call-it-something. */
        char buf[BUFSIZE];
        fcntl(fd, F_SETFL, fcntl(fd, F_GETFL) | O_NONBLOCK);

        fprintf(stderr,
          "Reading from something that does not support seeking (a pipe?),\n"
          "skipping to a position where reading would block\n"
        );

        while(read(fd, buf, BUFSIZE) > 0);
        if(errno != EAGAIN) {
            perror("read()");
            return 0;
        }
        fcntl(fd, F_SETFL, fcntl(fd, F_GETFL) & ~O_NONBLOCK);
    }

    return find_next_packet(fd);
}

static int usleep_ovcorr(struct timeval *req, long *skew) {
    /* The manual page for nanosleep(2) says this under the "BUGS" section:

      >The current implementation of nanosleep is  based  on  the
      >normal  kernel  timer mechanism, which has a resolution of
      >1/HZ s (i.e, 10 ms on Linux/i386 and 1 ms on Linux/Alpha).

    (*) for i386/2.4: HZ=100, i.e. 10 ms (10000 µs)
        for  all/2.6: HZ=1000, i.e. 1 ms (1000 µs)

    I do not claim that all 2.4 have HZ=100, especially since distributors
    like to change such. Just as likely, someone may set HZ back to 100 for
    a 2.6 one.

      >Therefore, nanosleep pauses always for at least the speci-
      >fied time, however it can take up to  10  ms  longer  than
      >specified  until  the  process becomes runnable again. [...]

      >As some applications  require  much  more  precise  pauses
      >(e.g.,  in  order to control some time-critical hardware),
      >nanosleep is also capable of short high-precision  pauses.
      >If  the process is scheduled under a real-time policy like
      >SCHED_FIFO or SCHED_RR, then pauses of up to 2 ms will  be
      >performed as busy waits with microsecond precision.

    Unfortunately, repeated busy waits lock out all other applications, so
    selecting another scheduling policy is inacceptable for ttyreplay, also
    because selecting another policy is only available to the superuser.

    Thus, I have designed a function that tries to cope with the longer delays
    by waiting a little shorter the next time when the previous call took
    longer. The minimal margin of course still is 1/100 (1/1000) s, why the
    algorithm splits up into multiple cases to make the best performance. They
    are explained below. */

    struct timeval start, stop;
    struct timespec nano_req;
    long long req_usec;
    long dur, over;
    int rv;

    /* The following if() block involves a "lot" of 64-bit calculations,
    which take more time on 32-bit archs. Even if my current AMD runs at
    approx. 30 billion instructions/second, it is only fair to add these few
    nanoseconds to the skew. To be non-plus-ultra fair, they are only added if
    we do not do an early return (case 2 & 3). */
    gettimeofday(&start, NULL);
    req_usec = tv2usec(req);

    if(req_usec > Opt.ovcorr) {
        /* If the user requests a delay which is greater than the minimal delay
        (calculated by calc_ovcorr()), we can simply take away some time from
        the request. */

        if(*skew + Opt.ovcorr <= req_usec) {
            /* If the accumulated skew time plus the minimal delay fits into
            the request, the request is reduced and the skew is zeroed. */
            usec2tv(req_usec -= *skew + Opt.ovcorr, req);
            *skew = 0;
        } else {
            /* There is more skew than the current request, so we can take away
            at most the requested time. (In simple language: We already paused
            enough in the past so we can entirely skip this pause.) */
            *skew -= req_usec;
            return 0;
        }
    } else if(*skew >= -Opt.ovcorr) {
        /* The code is the same as the above case, the intent too, but with a
        specialty: The skew can become negative (i.e. we paused too few) up to
        a specific degree (1/HZ s). */
        *skew -= req_usec;
        return 0;
    }

    /* If none of these three cases, or case 2 applies, nanosleep() will be
    called with the time request. */
    nano_req.tv_sec  = req->tv_sec;
    nano_req.tv_nsec = req->tv_usec * 1000,
    rv = nanosleep(&nano_req, NULL);
    gettimeofday(&stop, NULL);

    /* Calculate the actual duration of nanosleep(), and from that, the
    overhead (actual time minus wanted time) which is added to the skew. */
    dur = MICROSECOND * (stop.tv_sec - start.tv_sec) +
     (stop.tv_usec - start.tv_usec);
    // using req_usec is a little trick to save 64-bit calculations
    over = dur - req_usec;
    *skew += over;

    return rv;
}

static void tv_delta(const struct timeval *past, const struct timeval *now,
 struct timeval *dest) {
    /* Calculates the time difference between "past" and "now" and stores the
    result in "dest". All parameters in µs. */
    unsigned long sec = now->tv_sec - past->tv_sec;
    long acc = now->tv_usec - past->tv_usec;

    if(acc < 0) {
        // past: 1.5, now: 2.0, sec = 2 - 1 = 1, acc = 0 - 500000 = -500000;
        dest->tv_sec = --sec;
        dest->tv_usec = MICROSECOND + acc;
    } else {
        dest->tv_sec = sec;
        dest->tv_usec = acc;
    }

    return;
}

inline static unsigned long long tv2usec(const struct timeval *req) {
    return req->tv_sec * MICROSECOND + req->tv_usec;
}

inline static void usec2tv(unsigned long long usec, struct timeval *req) {
    req->tv_sec  = usec / MICROSECOND;
    req->tv_usec = usec % MICROSECOND;
    return;
}

//==[ End of file ]============================================================
