/*=============================================================================
ttyrpld - TTY replay daemon
user/rdsh.c - Shared functions for RPLD/INFOD/RPLCTL
  Copyright © Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004
  -- License restrictions apply (GPL v2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see the COPYING file
=============================================================================*/
#include <sys/socket.h>
#include <sys/types.h>
#include <errno.h>
#include <pthread.h>
#include <sched.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>

#include <libHX.h>
#include "dev.h"
#include "global.h"
#include "rdsh.h"
#define KERNEL_VERSION(a, b, c) (((a) << 16) | ((b) << 8) | (c))

// Functions
inline static int K_VERSION(void);
static void setup_kversion(void);

// External Variables
pthread_mutex_t Ttys_lock = PTHREAD_MUTEX_INITIALIZER;
struct HXbtree *Ttys = NULL;
struct Statmap_t Stats;
struct GOptmap_t GOpt = {
    .user_id    = -1,
    .syslog     = 0,
    .verbose    = 0,
    .infod_port = "/var/run/.rplinfod_socket",
    .ofmt       = "%u/%d.%t.%l", // -O
};

// Variables
static unsigned long k_version_data = 0;
static pthread_mutex_t k_version_lock = PTHREAD_MUTEX_INITIALIZER;

//-----------------------------------------------------------------------------
#if defined(__linux__)
static const char *m_idx = "pqrstuvwxyzabcdef", *s_idx = "0123456789abcdef";

char *G_devname_nm(uint32_t dev, char *buf, size_t count) {
    /* This function returns a simple name which can be integrated into
    rpld's log file name scheme. */
    register unsigned long major = K26_MAJOR(dev), minor = K26_MINOR(dev);
    *buf = '\0';

    switch(major) {
        case 2: // BSD pty masters
            snprintf(buf, count, "pty%c%c",
             m_idx[minor / 16], s_idx[minor % 16]);
            break;
        case 3: // BSD pty slaves
            snprintf(buf, count, "tty%c%c",
             m_idx[minor / 16], s_idx[minor % 16]);
            break;
        case 4:
            // Virtual console (0-63) and serial console (64-255)
            if(minor < 64) { snprintf(buf, count, "vc-%lu", minor); }
            else { snprintf(buf, count, "ttyS%lu", minor - 64); }
            break;
        case 5:
            if(minor == 1) { snprintf(buf, count, "console"); }
            break;
        case 43:
            snprintf(buf, count, "ttyI%lu", minor);
            break;
    }

    if(K_VERSION() >= KERNEL_VERSION(2, 6, 0)) {
        switch(major) {
            case 128: // Unix98 pty masters (got no real /dev entry)
                snprintf(buf, count, "ptm-%lu", minor);
                break;
            case 136: // Unix98 pty slaves
                snprintf(buf, count, "pts-%lu", minor);
                break;
        }
    } else {
        switch(major) {
            case 128 ... 135:
                snprintf(buf, count, "ptm-%lu",
                 COMPAT_MKDEV(major - 128, minor));
                break;
            case 136 ... 143:
                snprintf(buf, count, "pts-%lu",
                 COMPAT_MKDEV(major - 136, minor));
                break;
        }
    }

    if(*buf == '\0') { snprintf(buf, count, "[%lu:%lu]", major, minor); }
    return buf;
}

char *G_devname_fs(uint32_t dev, char *buf, size_t count) {
    /* This function returns a device filename generated from major/minor
    number, which is then used for stat(). */
    register unsigned long major = K26_MAJOR(dev), minor = K26_MINOR(dev);
    *buf = '\0';

    switch(major) {
        case 2: // BSD pty masters
            snprintf(buf, count, "/dev/pty%c%c",
             m_idx[minor / 16], s_idx[minor % 16]);
            break;
        case 3: // BSD pty slaves
            snprintf(buf, count, "/dev/tty%c%c",
             m_idx[minor / 16], s_idx[minor % 16]);
            break;
        case 4:
            if(minor < 64) { // Virtual console (0-63)
                snprintf(buf, count, "/dev/tty%lu", minor);
            } else { // Serial console (64-255)
                snprintf(buf, count, "/dev/ttyS%lu", minor - 64);
            }
            break;
        case 5:
            if(minor == 1) { snprintf(buf, count, "/dev/console"); }
            break;
        case 43:
            snprintf(buf, count, "/dev/ttyI%lu", minor);
            break;
    }

    if(K_VERSION() >= KERNEL_VERSION(2, 6, 0)) {
        switch(major) {
            /* Unix98 pty masters. They usually have no device nodes in /dev,
            and should instead be accessed over the multiplexer /dev/ptmx. */
            case 128: // Unix98 pty masters
                snprintf(buf, count, "/dev/ptmx");
                break;
            case 136: // Unix98 pty slaves
                snprintf(buf, count, "/dev/pts/%lu", minor);
                break;
        }
    } else {
        switch(major) {
            case 128 ... 135:
                snprintf(buf, count, "/dev/ptmx");
                break;
            case 136 ... 143: // Unix98 pty slaves
                snprintf(buf, count, "/dev/pts/%lu",
                 COMPAT_MKDEV(major - 136, minor));
                break;
        }
    }

    if(*buf == '\0') { return NULL; }
    return buf;
}

#elif defined(__FreeBSD__)
/* It is completely wicked. A lot of devices are assigned dynamic major numbers
(good idea but does not work in harmony with rpld ;-) Also disappointing is the
major number difference to Linux, as well as the [ms]_idx. */

static const char *m_idx = "pqrsPQRS",
 *s_idx = "0123456789abcdefghijklmnopqrstuv";

char *G_devname_nm(uint32_t dev, char *buf, size_t count) {
    register unsigned long major = K26_MAJOR(dev), minor = K26_MINOR(dev);
    *buf = '\0';

    switch(major) {
        case 0:
            snprintf(buf, count, "console");
            break;
        case 5:
            snprintf(buf, count, "tty%c%c",
             m_idx[minor / 16], s_idx[minor % 16]);
            break;
        case 6:
            snprintf(buf, count, "pty%c%c",
             m_idx[minor / 16], s_idx[minor % 16]);
            break;
    }

    if(*buf == '\0') { snprintf(buf, count, "[%lu:%lu]", major, minor); }
    return buf;
}

char *G_devname_fs(uint32_t dev, char *buf, size_t count) {
    register unsigned long major = K26_MAJOR(dev), minor = K26_MINOR(dev);
    *buf = '\0';

    switch(major) {
        case 0:
            snprintf(buf, count, "/dev/console");
            break;
        case 5:
            snprintf(buf, count, "/dev/tty%c%c",
             m_idx[minor / 16], s_idx[minor % 16]);
            break;
        case 6:
            snprintf(buf, count, "/dev/pty%c%c",
             m_idx[minor / 16], s_idx[minor % 16]);
            break;
    }

    if(*buf == '\0') { return NULL; }
    return buf;
}
#endif

struct tty *get_tty(uint32_t dev, int create) {
    struct HXbtree_node *ts;
    struct tty *ret = NULL, *tty;

    if((ret = HXbtree_get(Ttys, (void *)dev)) != NULL) { return ret; }
    if(!create || (tty = malloc(sizeof(struct tty))) == NULL) { return NULL; }

    tty->dev    = dev;
    tty->uid    = -1;
    tty->fd     = -1;
    tty->status = IFP_ACTIVATE;
    tty->in     = tty->out = 0;
    tty->file   = NULL;

    if((ts = HXbtree_add(Ttys, (void *)dev, tty)) == NULL) {
        free(tty);
        notify(LOG_ERR, _("%s: Memory allocation failure\n"), __FUNCTION__);
        return NULL;
    }

    return ts->data;
}

void log_close(struct tty *tty) {
    /* Close the logfile and release the tty struct if it does not have special
    options have set. */
    close(tty->fd);
    tty->fd = -1;
    if(tty->file != NULL) {
        free(tty->file);
        tty->file = NULL; // infod
    }

    if(tty->status != IFP_DEACTIVATE) {
        /* If the status is IFP_ACTIVATED, it is reinstantiated upon next
        get_tty(). If it is IFP_DEACTIVSES, it will change to IFP_ACTIVATED,
        as per definition. So we only need the data structure if
        IFP_DEACTIVATED is on. */
        HXbtree_del(Ttys, (void *)tty->dev);
        free(tty);
    }
    return;
}

void notify(int lv, const char *fmt, ...) {
    if(GOpt.verbose) {
        va_list argp;
        va_start(argp, fmt);
        fprintf(stderr, "\n");
        vfprintf(stderr, fmt, argp);
        va_end(argp);
        return; // do not print to syslog if we do to stdout
    }
    if(GOpt.syslog) {
        va_list argp;
        va_start(argp, fmt);
        vsyslog(lv, fmt, argp);
        va_end(argp);
        return;
    }
    return;
}

ssize_t send_wait(int fd, const void *buf, size_t count, int flags) {
    size_t rem = count;
    while(rem > 0) {
        ssize_t eax = send(fd, buf, rem, flags);
        if(eax < 0) { return -errno; }
        if(eax == rem) { break; }
        buf += eax;
        rem -= eax;
        sched_yield();
    }
    return count;
}

//-----------------------------------------------------------------------------
inline static int K_VERSION(void) {
    pthread_mutex_lock(&k_version_lock);
    if(k_version_data == 0) { setup_kversion(); }
    pthread_mutex_unlock(&k_version_lock);
    return k_version_data;
}

static void setup_kversion(void) {
    int x, y, z;
    FILE *fp;
    if((fp = fopen("/proc/version", "r")) == NULL) { return; }
    if(fscanf(fp, "%*s %*s %d.%d.%d", &x, &y, &z) != 3) { return; }
    fclose(fp);
    k_version_data = KERNEL_VERSION(x, y, z);
    return;
}

//==[ End of file ]============================================================
