/*=============================================================================
ttyrpld - TTY replay daemon
kernel/rpldev.c - Kernel interface for RPLD
  Copyright © Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004 - 2005
  -- License restrictions apply (GPL v2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see the COPYING file
=============================================================================*/
#include <asm/byteorder.h>
#include <asm/semaphore.h>
#include <asm/uaccess.h>
#include <linux/compiler.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/poll.h>
#include <linux/rpl.h>
#include <linux/string.h>
#include <linux/wait.h>
#include <linux/version.h>
#include <linux/vmalloc.h>
#include "../include/rpl_ioctl.h"
#include "../include/rpl_packet.h"

#ifndef __user
#    define __user
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
#    define KERNEL_2_4 1 // just for device number handling
#    define PTY_MASTER(tty) \
      (((tty)->driver->major >= UNIX98_PTY_MASTER_MAJOR && \
      (tty)->driver->major < UNIX98_PTY_MASTER_MAJOR + \
      UNIX98_PTY_MAJOR_COUNT) || (tty)->driver->major == PTY_MASTER_MAJOR)
#    define TTY_DEVNR(tty) \
      cpu_to_le32(mkdev_26(MAJOR(tty->device), MINOR(tty->device)))
#else
#    define PTY_MASTER(tty) \
      ((tty)->driver->major == UNIX98_PTY_MASTER_MAJOR || \
      (tty)->driver->major == PTY_MASTER_MAJOR)
#    define TTY_DEVNR(tty) \
      cpu_to_le32(mkdev_26(tty->driver->major, \
      tty->driver->minor_start + tty->index))
#endif

// Stage 2 functions
static int kio_init(struct tty_struct *, struct tty_struct *);
static int kio_open(struct tty_struct *, struct tty_struct *);
static int kio_read(const char __user *, size_t, struct tty_struct *);
static int kio_write(const char __user *, size_t, struct tty_struct *);
static int kio_close(struct tty_struct *, struct tty_struct *);
static int kio_deinit(struct tty_struct *, struct tty_struct *);
static int kio_ioctl(struct tty_struct *, struct tty_struct *, unsigned int,
  unsigned long);

// Stage 3 functions
static int uio_open(struct inode *, struct file *);
static ssize_t uio_read(struct file *, char __user *, size_t, loff_t *);
static loff_t uio_llseek(struct file *, loff_t, int);
static int uio_ioctl(struct inode *, struct file *, unsigned int,
 unsigned long);
static unsigned int uio_poll(struct file *, poll_table *);
static int uio_close(struct inode *, struct file *);

// Local functions
inline static ssize_t avail_R(void);
inline static ssize_t avail_W(void);
inline static unsigned int imin_uint(unsigned int, unsigned int);
inline static uint32_t mkdev_26(unsigned long, unsigned long);
static int mv_buffer(struct rpld_packet *, const void *, size_t);
inline static void mv_buffer2(const void *, size_t);
inline static void mv_to_user(char __user *, size_t);

// Variables
static DECLARE_WAIT_QUEUE_HEAD(Pull_queue);
static DECLARE_MUTEX(Buffer_lock);
static DECLARE_MUTEX(Open_lock);
static char *Buffer = NULL, *BufRP = NULL, *BufWP = NULL;
static size_t Bufsize = 32768;
static int Minor_nr = MISC_DYNAMIC_MINOR, Open_count = 0;

// Kernel module info (kmi) stuff
static struct file_operations kmi_fops = {
    .open    = uio_open,
    .read    = uio_read,
    .llseek  = uio_llseek,
    .ioctl   = uio_ioctl,
    .poll    = uio_poll,
    .release = uio_close,
    .owner   = THIS_MODULE,
};
static struct miscdevice kmi_miscinfo = {
    .minor = MISC_DYNAMIC_MINOR,
    .name  = "rpl",
    .fops  = &kmi_fops,
};

extern int __init init_module(void);
extern void __exit cleanup_module(void);
MODULE_DESCRIPTION("ttyrpld Kernel interface");
MODULE_AUTHOR("Jan Engelhardt <jengelh [at] linux01 gwdg de>");
MODULE_LICENSE("GPL v2");
MODULE_PARM_DESC(Bufsize, "Buffer size (default 32K)");
MODULE_PARM_DESC(Minor_nr, "Minor number to use (default: 255(=DYNAMIC))");
#ifdef KERNEL_2_4
MODULE_PARM(Bufsize, "i");
MODULE_PARM(Minor_nr, "i");
EXPORT_NO_SYMBOLS;
#else
module_param(Bufsize, uint, S_IRUGO);
module_param(Minor_nr, int, S_IRUGO);
#endif

//-----------------------------------------------------------------------------
int __init init_module(void) {
    int eax;
    kmi_miscinfo.minor = Minor_nr;
    if((eax = misc_register(&kmi_miscinfo)) != 0) {
        return eax;
    }
    Minor_nr = kmi_miscinfo.minor; // give it back to sysfs
    return 0;
}

void __exit cleanup_module(void) {
    misc_deregister(&kmi_miscinfo);
    return;
}

//-----------------------------------------------------------------------------
static int kio_init(struct tty_struct *tty, struct tty_struct *ctl) {
    struct rpld_packet p;

    if(PTY_MASTER(tty)) { return 0; }

    p.dev   = TTY_DEVNR(tty);
    p.size  = 0;
    p.event = EVT_INIT;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, NULL, 0);
}

static int kio_open(struct tty_struct *tty, struct tty_struct *ctl) {
    struct rpld_packet p;

    if(PTY_MASTER(tty)) { return 0; }

    p.dev   = TTY_DEVNR(tty);
    p.size  = 0;
    p.event = EVT_OPEN;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, NULL, 0);
}

static int kio_read(const char __user *buf, size_t count,
 struct tty_struct *tty)
{
    /* The data flow is a bit weird at first. kio_read() gets the data on its
    way between ttyDriver(master) -> /dev/stdin(slave), meaning this function
    is called when you hit the keyboard. */
    struct rpld_packet p;

    if(count == 0 || PTY_MASTER(tty)) { return 0; }

    p.dev   = TTY_DEVNR(tty);
    p.size  = cpu_to_le16(count);
    p.event = EVT_READ;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, buf, count);
}

static int kio_write(const char __user *buf, size_t count,
 struct tty_struct *tty)
{
    // Data flow: /dev/stdout(slave) -> tty driver(master)
    struct rpld_packet p;

    if(count == 0 || PTY_MASTER(tty)) { return 0; }

    p.dev   = TTY_DEVNR(tty);
    p.size  = cpu_to_le16(count);
    p.event = EVT_WRITE;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, buf, count);
}

static int kio_ioctl(struct tty_struct *tty, struct tty_struct *ctl,
 unsigned int cmd, unsigned long arg)
{
    struct rpld_packet p;
    uint32_t cmd32;

    if(PTY_MASTER(tty)) { return 0; }

    cmd32   = cmd;
    p.dev   = TTY_DEVNR(tty);
    p.size  = cpu_to_le16(sizeof(cmd32));
    p.event = EVT_IOCTL;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, &cmd32, sizeof(cmd32));
}

static int kio_close(struct tty_struct *tty, struct tty_struct *other) {
    struct rpld_packet p;

    if(PTY_MASTER(tty)) { return 0; }

    p.dev   = TTY_DEVNR(tty);
    p.size  = 0;
    p.event = EVT_CLOSE;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, NULL, 0);
}

static int kio_deinit(struct tty_struct *tty, struct tty_struct *other) {
    struct rpld_packet p;

    if(PTY_MASTER(tty)) { tty = other; }

    p.dev   = TTY_DEVNR(tty);
    p.size  = 0;
    p.event = EVT_DEINIT;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, NULL, 0);
}

//-----------------------------------------------------------------------------
static int uio_open(struct inode *inode, struct file *filp) {
    // This one is called when the device node has been opened.
    if(inode != NULL) {
        inode->i_mtime = CURRENT_TIME;
        inode->i_mode &= ~(S_IWUGO | S_IXUGO);
    }

    /* The RPL device can only be opened once, since otherwise, different
    packets could go to different readers. It is already "dangerous" with
    fork() which can kinda step over this. Yet safe against spurious
    open()s. */

    down(&Open_lock);
    if(Open_count) {
        up(&Open_lock);
        return -EBUSY;
    }
    ++Open_count;
    up(&Open_lock);

    down(&Buffer_lock);
    Buffer = __vmalloc(Bufsize, GFP_KERNEL | __GFP_HIGHMEM, PAGE_KERNEL);
    if(Buffer == NULL) {
        up(&Buffer_lock);
        up(&Open_lock);
        return -ENOMEM;
    }
    BufRP = BufWP = Buffer;

    /* Update links. I do it here 'cause I do not want memory copying (from the
    tty driver to the buffer) when there is no one to read. */
    rpl_qinit   = kio_init;
    rpl_qopen   = kio_open;
    rpl_qread   = kio_read;
    rpl_qwrite  = kio_write;
    rpl_qclose  = kio_close;
    rpl_qioctl  = kio_ioctl;
    rpl_qdeinit = kio_deinit;
    up(&Buffer_lock);

    /* The inode's times are changed as follows:
            Access Time: if data is read from the device
      Inode Change Time: when the device is successfully opened
      Modification Time: whenever the device is opened
    */
    if(inode != NULL) { inode->i_ctime = CURRENT_TIME; }
    return 0;
}

static ssize_t uio_read(struct file *filp, char __user *buf,
 size_t count, loff_t *ppos)
{
    // Nothing read, nothing done
    if(count == 0) { return 0; }

    // Must sleep as long as there is no data
    if(down_interruptible(&Buffer_lock)) { return -ERESTARTSYS; }
    while(BufRP == BufWP) {
        up(&Buffer_lock);
        if(filp->f_flags & O_NONBLOCK) { return -EAGAIN; }
        if(wait_event_interruptible(Pull_queue, (BufRP != BufWP))) {
            return -ERESTARTSYS;
        }
        if(down_interruptible(&Buffer_lock)) { return -ERESTARTSYS; }
    }

    // Data is available, so give it to the user
    count = imin_uint(count, avail_R());
    mv_to_user(buf, count);

    up(&Buffer_lock);
    filp->f_dentry->d_inode->i_atime = CURRENT_TIME;
    return count;
}

static loff_t uio_llseek(struct file *filp, loff_t offset, int origin) {
    /* Since you can not seek in the circular buffer, the only purpose of
    lseek() is to skip some bytes and "manually" advance the read pointer RP
    (BufRP). Thus, the only accepted origin is SEEK_CUR (1), or SEEK_END (2)
    with offset==0. */

    if(origin == 2 && offset == 0) {
        if(down_interruptible(&Buffer_lock)) { return -ERESTARTSYS; }
        BufRP = BufWP;
        up(&Buffer_lock);
        filp->f_dentry->d_inode->i_atime = CURRENT_TIME;
        return 0;
    }

    if(origin != 1) { return -ESPIPE; }
    if(offset == 0) { return 0; }

    if(down_interruptible(&Buffer_lock)) { return -ERESTARTSYS; }
    BufRP = Buffer + (BufRP - Buffer + imin_uint(offset, avail_R())) % Bufsize;
    WARN_ON(BufRP >= Buffer + Bufsize);
    if(BufRP >= Buffer + Bufsize) { 
        WARN_ON(1);
        BufRP = BufWP = 0;
    }

    up(&Buffer_lock);
    filp->f_dentry->d_inode->i_atime = CURRENT_TIME;
    return 0;
}

static int uio_ioctl(struct inode *inode, struct file *filp, unsigned int cmd,
 unsigned long arg)
{
    if(_IOC_TYPE(cmd) != RPL_IOC_MAGIC) { return -ENOTTY; }

    switch(cmd) {
        case RPL_IOC_GETBUFSIZE:
            return Bufsize;
        case RPL_IOC_GETRAVAIL:
            return avail_R();
        case RPL_IOC_GETWAVAIL:
            return avail_W();
        default:
            return -ENOTTY;
    }
    return 0;
}

static unsigned int uio_poll(struct file *filp, poll_table *wait) {
    poll_wait(filp, &Pull_queue, wait);
    return POLLIN | POLLRDNORM;
}

static int uio_close(struct inode *inode, struct file *filp) {
    rpl_qinit   = NULL;
    rpl_qopen   = NULL;
    rpl_qread   = NULL;
    rpl_qwrite  = NULL;
    rpl_qioctl  = NULL;
    rpl_qclose  = NULL;
    rpl_qdeinit = NULL;
    down(&Buffer_lock);
    vfree(Buffer);
    Buffer = NULL;
    up(&Buffer_lock);
    --Open_count;
    return 0;
}

//-----------------------------------------------------------------------------
inline static ssize_t avail_R(void) {
    // Return the number of available bytes to read
    if(BufWP >= BufRP) { return BufWP - BufRP; }
    return BufWP + Bufsize - BufRP;
}

inline static ssize_t avail_W(void) {
    // Return the number of available bytes to write
    if(BufWP >= BufRP) { return BufRP + Bufsize - BufWP - 1; }
    return BufRP - BufWP - 1;
}

inline static unsigned int imin_uint(unsigned int a, unsigned int b) {
    return a < b ? a : b;
}

inline static uint32_t mkdev_26(unsigned long maj, unsigned long min) {
    return (maj << 20) | (min & 0xFFFFF);
}

static int mv_buffer(struct rpld_packet *p, const void *buf, size_t count) {
    /* Copy the contents the tty driver received to our circulary buffer and
    also add a header (rpld_packet) to it so that the userspace daemon can
    recognize the type. */
    if(down_interruptible(&Buffer_lock)) { return -ERESTARTSYS; }
    if(avail_W() < sizeof(struct rpld_packet) + count) {
        up(&Buffer_lock);
        return -ENOSPC;
    }

    mv_buffer2(p, sizeof(struct rpld_packet));
    if(count > 0) { mv_buffer2(buf, count); }

    up(&Buffer_lock);
    wake_up(&Pull_queue);
    return count;
}

inline static void mv_buffer2(const void *src, size_t count) {
    /* This function is responsible for copying (a specific amount of
    arbitrary data) into the circulary buffer. (Taking the wrappage into
    account!) The parent function -- mv_buffer() in this case -- must make sure
    there is enough room. */
    size_t x = Buffer + Bufsize - BufWP;

    if(count < x) {
        memcpy(BufWP, src, count);
        BufWP += count;
    } else {
        memcpy(BufWP, src, x);
        memcpy(Buffer, src + x, count - x);
        BufWP = Buffer + count - x;
    }

    return;
}

inline static void mv_to_user(char __user *dest, size_t count) {
    /* mv_to_user() reads COUNT bytes from the circulary buffer and puts it
    into userspace memory. The caller must make sure that COUNT is at most
    avail_R(). */
    size_t x = Buffer + Bufsize - BufRP;

    /* Since this operation might block due to userspace page faults, the
    userspace daemon should really have DEST memory-locked. */
    if(count < x) {
        copy_to_user(dest, BufRP, count);
        BufRP += count;
    } else {
        copy_to_user(dest, BufRP, x);
        copy_to_user(dest + x, Buffer, count - x);
        BufRP = Buffer + count - x;
    }
    return;
}

//==[ End of file ]============================================================
