/*=============================================================================
ttyrpld - TTY replay daemon
freebsd-5.3/rpldev.c - Stage 1 RPL interface
  Copyright © Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004 - 2005
  -- License restrictions apply (GPL v2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  -- For details see the COPYING file
=============================================================================*/
#include <sys/param.h>
#include <sys/conf.h>
#include <sys/endian.h>
#include <sys/errno.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/module.h>
#include <sys/mutex.h>
#include <sys/proc.h>
#include <sys/systm.h>
#include <sys/tty.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/rpl.h>
#include "../include/rpl_ioctl.h"
#include "../include/rpl_packet.h"

#define IO_NDELAY 0x10 // sys/vnode.h -- inclusion problems
#define PTY_MASTER(tty) (major(tty->t_dev) == 6)
#define TTY_DEVNR(tty) htole32(mkdev_26(major(tty->t_dev), minor(tty->t_dev)))
#define RET_IF_INACT if(!kmi_usecount) { return 0; }

// Module stuff
static int kmd_event(module_t, int, void *);
static int kmd_load(void);
static int kmd_unload(void);

// Stage 2 functions
static int kio_init(struct cdev *, struct tty *);
static int kio_open(struct cdev *, struct tty *);
static int kio_read(const char *, int, struct tty *);
static int kio_write(const char *, int, struct tty *);
static int kio_ioctl(struct tty *, u_long, void *);
static int kio_close(struct tty *);
static int kio_deinit(struct tty *);

// Stage 3 functions
static int uio_open(struct cdev *, int, int, struct thread *);
static int uio_read(struct cdev *, struct uio *, int);
static int uio_close(struct cdev *, int, int, struct thread *);

// Local functions
inline static ssize_t avail_R(void);
inline static ssize_t avail_W(void);
inline static unsigned int min_uint(unsigned int, unsigned int);
inline static uint32_t mkdev_26(unsigned long, unsigned long);
static int mv_buffer(struct rpld_packet *, const void *, size_t);
inline static void mv_buffer2(const void *, size_t);
inline static int mv_to_user(struct uio *, size_t);

// Variables
static MALLOC_DEFINE(Buffer_malloc, "rpldev", "rpldev ring buffer");
static struct mtx Buffer_lock, Open_lock;
static char *Buffer = NULL, *BufRP = NULL, *BufWP = NULL;
static size_t Bufsize = 32768;
static int Open_count = 0, Pull_queue;

// Kernel module info (kmi) stuff
static int kmi_usecount = 0;
static struct cdev *kmi_node;
static struct cdevsw kmi_fops = {
    .d_version = D_VERSION,
    .d_name    = "rpldev",
    .d_maj     = MAJOR_AUTO,
    .d_open    = uio_open,
    .d_read    = uio_read,
    .d_close   = uio_close,
};
static moduledata_t kmi_rpldev = {
    .name   = "rpldev",
    .evhand = kmd_event,
};

DECLARE_MODULE(rpldev, kmi_rpldev, SI_SUB_DRIVERS, SI_ORDER_MIDDLE);

//-----------------------------------------------------------------------------
static int kmd_event(module_t mp, int type, void *data) {
    int eax;
    switch(type) {
        case MOD_LOAD:
            if((eax = kmd_load()) != 0) { kmd_unload(); }
            return eax;
            break;
        case MOD_UNLOAD:
            return kmd_unload();
            break;
    }
    return EOPNOTSUPP;
}

static int kmd_load(void) {
    mtx_init(&Buffer_lock, "rpldev", NULL, MTX_DEF);
    mtx_init(&Open_lock, "rpldev", NULL, MTX_DEF);
    if((kmi_node = make_dev(&kmi_fops, 0, UID_ROOT, 0, 0600, "rpl")) == NULL) {
        return ENOMEM;
    }
    return 0;
}

static int kmd_unload(void) {
    if(kmi_usecount) { return EBUSY; }
    if(kmi_node != NULL) { destroy_dev(kmi_node); }
    mtx_destroy(&Buffer_lock);
    mtx_destroy(&Open_lock);
    return 0;
}

//-----------------------------------------------------------------------------
static int kio_init(struct cdev *cd, struct tty *tty) {
    struct rpld_packet p;

    RET_IF_INACT;
    if(PTY_MASTER(tty)) { return 0; }

    p.dev   = TTY_DEVNR(tty);
    p.size  = 0;
    p.event = EVT_INIT;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, NULL, 0);
}

static int kio_open(struct cdev *cd, struct tty *tty) {
    struct rpld_packet p;

    RET_IF_INACT;
    if(PTY_MASTER(tty)) { return 0; }

    p.dev   = TTY_DEVNR(tty);
    p.size  = 0;
    p.event = EVT_OPEN;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, NULL, 0);
}

static int kio_read(const char *buf, int count, struct tty *tty) {
    struct rpld_packet p;

    RET_IF_INACT;
    if(count == 0 || PTY_MASTER(tty)) { return 0; }

    p.dev   = TTY_DEVNR(tty);
    p.size  = htole16(count);
    p.event = EVT_READ;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, buf, count);
}

static int kio_write(const char *buf, int count, struct tty *tty) {
    struct rpld_packet p;

    RET_IF_INACT;
    if(count == 0 || PTY_MASTER(tty)) { return 0; }

    p.dev   = TTY_DEVNR(tty);
    p.size  = htole16(count);
    p.event = EVT_WRITE;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, buf, count);
}

static int kio_ioctl(struct tty *tty, u_long cmd, void *arg) {
    struct rpld_packet p;
    uint32_t cmd32;

    RET_IF_INACT;
    if(PTY_MASTER(tty)) { return 0; }

    cmd32   = cmd;
    p.dev   = TTY_DEVNR(tty);
    p.size  = htole16(sizeof(cmd32));
    p.event = EVT_IOCTL;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, &cmd32, sizeof(cmd32));
}

static int kio_close(struct tty *tty) {
    struct rpld_packet p;

    RET_IF_INACT;
    if(PTY_MASTER(tty)) { return 0; }

    p.dev   = TTY_DEVNR(tty);
    p.size  = 0;
    p.event = EVT_CLOSE;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, NULL, 0);
}

static int kio_deinit(struct tty *tty) {
    struct rpld_packet p;

    RET_IF_INACT;
    if(PTY_MASTER(tty)) { return 0; }

    p.dev   = TTY_DEVNR(tty);
    p.size  = 0;
    p.event = EVT_DEINIT;
    p.magic = MAGIC_SIG;
    return mv_buffer(&p, NULL, 0);
}

//-----------------------------------------------------------------------------
static int uio_open(struct cdev *cd, int flag, int mode, struct thread *th) {
    mtx_lock(&Open_lock);
    if(Open_count) {
        mtx_unlock(&Open_lock);
        return EBUSY;
    }
    ++kmi_usecount;
    ++Open_count;
    mtx_unlock(&Open_lock);

    mtx_lock(&Buffer_lock);
    if((Buffer = malloc(Bufsize, Buffer_malloc, M_WAITOK)) == NULL) {
        mtx_unlock(&Buffer_lock);
        --kmi_usecount;
        return ENOMEM;
    }

    BufRP = BufWP = Buffer;
    rpl_qinit   = kio_init;
    rpl_qopen   = kio_open;
    rpl_qread   = kio_read;
    rpl_qwrite  = kio_write;
    rpl_qioctl  = kio_ioctl;
    rpl_qclose  = kio_close;
    rpl_qdeinit = kio_deinit;
    mtx_unlock(&Buffer_lock);
    return 0;
}

static int uio_read(struct cdev *cd, struct uio *uio, int flags) {
    size_t count;
    int eax;

    mtx_lock(&Buffer_lock);
    while(BufRP == BufWP) {
        mtx_unlock(&Buffer_lock);
        if(flags & IO_NDELAY) { return EWOULDBLOCK; }
        if((eax = tsleep(&Pull_queue, PCATCH, "rpldev", 0)) != 0) {
            return eax;
        }
        mtx_lock(&Buffer_lock);
    }
    count = min_uint(uio->uio_resid, avail_R());
    eax = mv_to_user(uio, count);
    mtx_unlock(&Buffer_lock);
    return eax;
}

static int uio_close(struct cdev *cd, int flags, int fmt, struct thread *th) {
    rpl_qinit   = NULL;
    rpl_qopen   = NULL;
    rpl_qread   = NULL;
    rpl_qwrite  = NULL;
    rpl_qioctl  = NULL;
    rpl_qclose  = NULL;
    rpl_qdeinit = NULL;
    mtx_lock(&Buffer_lock);
    free(Buffer, Buffer_malloc);
    mtx_unlock(&Buffer_lock);
    --Open_count;
    --kmi_usecount;
    return 0;
}

//-----------------------------------------------------------------------------
inline static ssize_t avail_R(void) {
    if(BufWP >= BufRP) { return BufWP - BufRP; }
    return BufWP + Bufsize - BufRP;
}

inline static ssize_t avail_W(void) {
    if(BufWP >= BufRP) { return BufRP + Bufsize - BufWP - 1; }
    return BufRP - BufWP - 1;
}

inline static unsigned int min_uint(unsigned int a, unsigned int b) {
    return a < b ? a : b;
}

inline static uint32_t mkdev_26(unsigned long maj, unsigned long min) {
    return (maj << 20) | (min & 0xFFFFF);
}

static int mv_buffer(struct rpld_packet *p, const void *buf, size_t count) {
    mtx_lock(&Buffer_lock);
    if(avail_W() < sizeof(struct rpld_packet) + count) {
        mtx_unlock(&Buffer_lock);
        return ENOSPC;
    }

    mv_buffer2(p, sizeof(struct rpld_packet));
    if(count > 0) { mv_buffer2(buf, count); }
    mtx_unlock(&Buffer_lock);
    wakeup_one(&Pull_queue);
    return count;
}

inline static void mv_buffer2(const void *src, size_t count) {
    size_t x = Buffer + Bufsize - BufWP;

    if(count < x) {
        memcpy(BufWP, src, count);
        BufWP += count;
    } else {
        memcpy(BufWP, src, x);
        memcpy(Buffer, src + x, count - x);
        BufWP = Buffer + count - x;
    }

    return;
}

inline static int mv_to_user(struct uio *uio, size_t count) {
    size_t x = Buffer + Bufsize - BufRP;
    int eax;

    if(count < x) {
        if((eax = uiomove(BufRP, count, uio))) { return eax; }
        BufRP += count;
    } else {
        if((eax = uiomove(BufRP, x, uio)) ||
         (eax = uiomove(Buffer, count - x, uio))) {
            return eax;
        }
        BufRP = Buffer + count - x;
    }

    return 0;
}

//==[ End of file ]============================================================
