/*=============================================================================
ttyrpld - TTY replay daemon
freebsd-5.3/rpldev.c - Kernel interface for RPLD
  Copyright © Jan Engelhardt <jengelh [at] linux01 gwdg de>, 2004 - 2006
  -- License restrictions apply (GPL v2)

  This file is part of ttyrpld.
  ttyrpld is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; however ONLY version 2 of the License.

  ttyrpld is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program kit; if not, write to:
  Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA

  -- For details, see the file named "LICENSE.GPL2"
=============================================================================*/
#include <sys/param.h>
#include <sys/conf.h>
#include <sys/endian.h>
#include <sys/errno.h>
#include <sys/ioccom.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/module.h>
#include <sys/mutex.h>
#include <sys/proc.h>
#include <sys/syslog.h>
#include <sys/systm.h>
#include <sys/time.h>
#include <sys/tty.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/km_rpldev.h>
#include "../include/rpl_ioctl.h"
#include "../include/rpl_packet.h"

#define IO_NDELAY      0x10 // from sys/vnode.h -- inclusion problems
#define SKIP_PTM(tty)  if(major(tty->t_dev) == 6) return 0;
#define TTY_DEVNR(tty) htole32(mkdev_26(major(tty->t_dev), minor(tty->t_dev)))
#define RET_IF_INACT   if(!kmi_usecount) return 0;

// Module stuff
static int kmd_event(module_t, int, void *);
static int kmd_load(void);
static int kmd_unload(void);

// Stage 2 functions
static int krpl_init(struct cdev *, struct tty *);
static int krpl_open(struct cdev *, struct tty *);
static int krpl_read(const char *, int, struct tty *);
static int krpl_write(const char *, int, struct tty *);
//static int krpl_ioctl(struct tty *, u_long, void *);
static int krpl_close(struct tty *);
static int krpl_deinit(struct tty *);

// Stage 3 functions
static int urpl_open(struct cdev *, int, int, struct thread *);
static int urpl_read(struct cdev *, struct uio *, int);
static int urpl_ioctl(struct cdev *, u_long, caddr_t, int, struct thread *);
static int urpl_close(struct cdev *, int, int, struct thread *);

// Local functions
static inline ssize_t avail_R(void);
static inline ssize_t avail_W(void);
static inline void fill_time(struct timeval *);
static inline unsigned int min_uint(unsigned int, unsigned int);
static inline uint32_t mkdev_26(unsigned long, unsigned long);
static int mv_buffer(struct rpldev_packet *, const void *, size_t);
static inline void mv_buffer2(const void *, size_t);
static inline int mv_to_user(struct uio *, size_t);

// Variables
static MALLOC_DEFINE(Buffer_malloc, "rpldev", "rpldev ring buffer");
static struct mtx Buffer_lock, Open_lock;
static char *Buffer = NULL, *BufRP = NULL, *BufWP = NULL;
static size_t Bufsize = 32 * 1024;
static int Open_count = 0;

// Kernel module info stuff
static int kmi_usecount = 0;
static struct cdev *kmi_node;
static struct cdevsw kmi_fops = {
    .d_version = D_VERSION,
    .d_name    = "rpldev",
    .d_maj     = MAJOR_AUTO,
    .d_open    = urpl_open,
    .d_read    = urpl_read,
    .d_ioctl   = urpl_ioctl,
    .d_close   = urpl_close,
};
static moduledata_t kmi_rpldev = {
    .name   = "rpldev",
    .evhand = kmd_event,
};

DECLARE_MODULE(rpldev, kmi_rpldev, SI_SUB_DRIVERS, SI_ORDER_MIDDLE);

//-----------------------------------------------------------------------------
static int kmd_event(module_t mp, int type, void *data) {
    int eax;
    switch(type) {
        case MOD_LOAD:
            if((eax = kmd_load()) != 0)
                kmd_unload();
            return eax;
        case MOD_UNLOAD:
            return kmd_unload();
    }
    return EOPNOTSUPP;
}

static int kmd_load(void) {
    mtx_init(&Buffer_lock, "rpldev", NULL, MTX_DEF);
    mtx_init(&Open_lock, "rpldev", NULL, MTX_DEF);
    if((kmi_node = make_dev(&kmi_fops, 0, UID_ROOT, 0, 0600, "rpl")) == NULL)
        return ENOMEM;
    return 0;
}

static int kmd_unload(void) {
    if(kmi_usecount)
        return EBUSY;
    if(kmi_node != NULL)
        destroy_dev(kmi_node);
    mtx_destroy(&Buffer_lock);
    mtx_destroy(&Open_lock);
    return 0;
}

//-----------------------------------------------------------------------------
static int krpl_init(struct cdev *cd, struct tty *tty) {
    struct rpldev_packet p;

    RET_IF_INACT;
    SKIP_PTM(tty);

    p.dev   = TTY_DEVNR(tty);
    p.size  = 0;
    p.event = EVT_INIT;
    p.magic = MAGIC_SIG;
    fill_time(&p.time);
    return mv_buffer(&p, NULL, 0);
}

static int krpl_open(struct cdev *cd, struct tty *tty) {
    struct rpldev_packet p;

    RET_IF_INACT;
    SKIP_PTM(tty);

    p.dev   = TTY_DEVNR(tty);
    p.size  = 0;
    p.event = EVT_OPEN;
    p.magic = MAGIC_SIG;
    fill_time(&p.time);
    return mv_buffer(&p, NULL, 0);
}

static int krpl_read(const char *buf, int count, struct tty *tty) {
    struct rpldev_packet p;

    RET_IF_INACT;
    SKIP_PTM(tty);
    if(count == 0) return 0;

    p.dev   = TTY_DEVNR(tty);
    p.size  = htole16(count);
    p.event = EVT_READ;
    p.magic = MAGIC_SIG;
    fill_time(&p.time);
    return mv_buffer(&p, buf, count);
}

static int krpl_write(const char *buf, int count, struct tty *tty) {
    struct rpldev_packet p;

    RET_IF_INACT;
    SKIP_PTM(tty);
    if(count == 0) return 0;

    p.dev   = TTY_DEVNR(tty);
    p.size  = htole16(count);
    p.event = EVT_WRITE;
    p.magic = MAGIC_SIG;
    fill_time(&p.time);
    return mv_buffer(&p, buf, count);
}

/*
static int krpl_ioctl(struct tty *tty, u_long cmd, void *arg) {
    struct rpldev_packet p;
    uint32_t cmd32;

    RET_IF_INACT;
    SKIP_PTM(tty);

    cmd32   = cmd;
    p.dev   = TTY_DEVNR(tty);
    p.size  = htole16(sizeof(cmd32));
    p.event = EVT_IOCTL;
    p.magic = MAGIC_SIG;
    fill_time(&p.time);
    return mv_buffer(&p, &cmd32, sizeof(cmd32));
}
*/

static int krpl_close(struct tty *tty) {
    struct rpldev_packet p;

    RET_IF_INACT;
    SKIP_PTM(tty);

    p.dev   = TTY_DEVNR(tty);
    p.size  = 0;
    p.event = EVT_CLOSE;
    p.magic = MAGIC_SIG;
    fill_time(&p.time);
    return mv_buffer(&p, NULL, 0);
}

static int krpl_deinit(struct tty *tty) {
    struct rpldev_packet p;

    RET_IF_INACT;
    SKIP_PTM(tty);

    p.dev   = TTY_DEVNR(tty);
    p.size  = 0;
    p.event = EVT_DEINIT;
    p.magic = MAGIC_SIG;
    fill_time(&p.time);
    return mv_buffer(&p, NULL, 0);
}

//-----------------------------------------------------------------------------
static int urpl_open(struct cdev *cd, int flag, int mode, struct thread *th) {
    mtx_lock(&Open_lock);
    if(Open_count) {
        mtx_unlock(&Open_lock);
        return EBUSY;
    }
    ++kmi_usecount;
    ++Open_count;
    mtx_unlock(&Open_lock);

    mtx_lock(&Buffer_lock);
    if((Buffer = malloc(Bufsize, Buffer_malloc, M_WAITOK)) == NULL) {
        mtx_unlock(&Buffer_lock);
        --kmi_usecount;
        --Open_count;
        return ENOMEM;
    }

    BufRP = BufWP = Buffer;
    rpl_qinit   = krpl_init;
    rpl_qopen   = krpl_open;
    rpl_qread   = krpl_read;
    rpl_qwrite  = krpl_write;
    /* ioctl reporting is deactivated, because 1) a _lot_ more ioctls are
    generated in *BSD compared to Linux, 2) ioctls are not yet interpreted by
    userspace apps (ttyreplay). */
    //rpl_qioctl  = krpl_ioctl;
    rpl_qclose  = krpl_close;
    rpl_qdeinit = krpl_deinit;
    mtx_unlock(&Buffer_lock);
    return 0;
}

static int urpl_read(struct cdev *cd, struct uio *uio, int flags) {
    size_t count;
    int eax;

    mtx_lock(&Buffer_lock);
    while(BufRP == BufWP) {
        mtx_unlock(&Buffer_lock);
        if(flags & IO_NDELAY)
            return EWOULDBLOCK;
        if((eax = tsleep(&Buffer, PCATCH, "rpldev", 0)) != 0)
            return eax;
        mtx_lock(&Buffer_lock);
    }

    count = min_uint(uio->uio_resid, avail_R());
    eax = mv_to_user(uio, count);
    mtx_unlock(&Buffer_lock);
    return eax;
}

static int urpl_ioctl(struct cdev *cd, u_long cmd, caddr_t data, int flags,
 struct thread *th)
{
    /* Gah, it does not even support multiple return values >:-( so I need to
    hack around with data. */
    size_t *ptr = (void *)data;
    if(IOCGROUP(cmd) != RPL_IOC_MAGIC)
        return ENOTTY;
    if(ptr == NULL)
        return EFAULT;

    switch(cmd) {
        case RPL_IOC_GETBUFSIZE:
            *ptr = Bufsize;
            return 0;
        case RPL_IOC_GETRAVAIL:
            *ptr = avail_R();
            return 0;
        case RPL_IOC_GETWAVAIL:
            *ptr = avail_W();
            return 0;
        case RPL_IOC_IDENTIFY:
            *ptr = 0xC0FFEE;
            return 0;
        case RPL_IOC_SEEK:
            mtx_lock(&Buffer_lock);
            BufRP = Buffer + (BufRP - Buffer +
                    min_uint(*ptr, avail_R())) % Bufsize;
            if(BufRP >= Buffer + Bufsize) { 
                log(LOG_ERR, "BufRP overflow (should not happen!) -"
                 " resetting BufRP/BufWP\n");
                BufRP = BufWP = 0;
            }
            mtx_unlock(&Buffer_lock);
            return 0;
        case RPL_IOC_FLUSH:
            mtx_lock(&Buffer_lock);
            BufRP = BufWP;
            mtx_unlock(&Buffer_lock);
            return 0;
    }
    return ENOTTY;
}

static int urpl_close(struct cdev *cd, int flags, int fmt, struct thread *th) {
    rpl_qinit   = NULL;
    rpl_qopen   = NULL;
    rpl_qread   = NULL;
    rpl_qwrite  = NULL;
    rpl_qioctl  = NULL;
    rpl_qclose  = NULL;
    rpl_qdeinit = NULL;
    mtx_lock(&Buffer_lock);
    free(Buffer, Buffer_malloc);
    mtx_unlock(&Buffer_lock);
    --Open_count;
    --kmi_usecount;
    return 0;
}

//-----------------------------------------------------------------------------
static inline ssize_t avail_R(void) {
    if(BufWP >= BufRP)
        return BufWP - BufRP;
    return BufWP + Bufsize - BufRP;
}

static inline ssize_t avail_W(void) {
    if(BufWP >= BufRP)
        return BufRP + Bufsize - BufWP - 1;
    return BufRP - BufWP - 1;
}

static inline void fill_time(struct timeval *tv) {
    microtime(tv);

    if(sizeof(tv->tv_sec) == sizeof(uint32_t))
        tv->tv_sec = htole32(tv->tv_sec);
    else if(sizeof(tv->tv_sec) == sizeof(uint64_t))
        tv->tv_sec = htole64(tv->tv_sec);

    if(sizeof(tv->tv_usec) == sizeof(uint32_t))
        tv->tv_usec = htole32(tv->tv_usec);
    else if(sizeof(tv->tv_usec) == sizeof(uint64_t))
        tv->tv_usec = htole64(tv->tv_usec);
    return;
}

static inline unsigned int min_uint(unsigned int a, unsigned int b) {
    return (a < b) ? a : b;
}

static inline uint32_t mkdev_26(unsigned long maj, unsigned long min) {
    return (maj << 20) | (min & 0xFFFFF);
}

static int mv_buffer(struct rpldev_packet *p, const void *buf, size_t count) {
    mtx_lock(&Buffer_lock);
    if(avail_W() < sizeof(struct rpldev_packet) + count) {
        mtx_unlock(&Buffer_lock);
        return ENOSPC;
    }

    mv_buffer2(p, sizeof(struct rpldev_packet));
    if(count > 0)
        mv_buffer2(buf, count);
    mtx_unlock(&Buffer_lock);
    wakeup_one(&Buffer);
    return count;
}

static inline void mv_buffer2(const void *src, size_t count) {
    size_t x = Buffer + Bufsize - BufWP;

    if(count < x) {
        memcpy(BufWP, src, count);
        BufWP += count;
    } else {
        memcpy(BufWP, src, x);
        memcpy(Buffer, src + x, count - x);
        BufWP = Buffer + count - x;
    }

    return;
}

static inline int mv_to_user(struct uio *uio, size_t count) {
    size_t x = Buffer + Bufsize - BufRP;
    int eax;

    if(count < x) {
        if((eax = uiomove(BufRP, count, uio)))
            return eax;
        BufRP += count;
    } else {
        if((eax = uiomove(BufRP, x, uio)) ||
         (eax = uiomove(Buffer, count - x, uio)))
            return eax;
        BufRP = Buffer + count - x;
    }

    return 0;
}

//=============================================================================
